# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dynamodb_session_flask']

package_data = \
{'': ['*']}

install_requires = \
['Flask>=2.0.0,<3.0.0', 'dynamodb-session-web>=0.2.4,<0.3.0']

setup_kwargs = {
    'name': 'dynamodb-session-flask',
    'version': '0.5.0',
    'description': 'A session implementation for Flask using DynamoDB as a backing store and OWASP best practices for session management.',
    'long_description': '# dynamodb-session-flask\n\nAn implementation of a Flask session using DynamoDB as backend storage. This project was built on \n[dynamodb-session-web](https://github.com/JCapriotti/dynamodb-session-web), but with support for the Flask framework.\n\nIn addition to the [OWASP Session Management](https://cheatsheetseries.owasp.org/cheatsheets/Session_Management_Cheat_Sheet.html) \nbest practices implemented in `dynamodb-session-web`, this project has additional support for these best practices:\n\n* Non-descript session ID name - Defaults to `id` for cookies, and `x-id` for headers. \n  * Side-Comment - isn\'t a non-descript suggestion for a name actually descriptive?\n* Cookie setting defaults:\n  - [X] Secure = True\n  - [X] HttpOnly = True\n  - [X] SameSite = Strict\n  - [ ] Domain and Path - Must set these yourself\n* ID Exchange\n  - [X] Accepted session ID mechanism (i.e. cookie vs header) is enforced. That is, user cannot submit session IDs \n        through a header if cookie is expected.\n\n## Usage\n\nRequires a DynamoDB table named `app_session` (can be changed in settings). \n\nHere\'s an example table creation statement:\n\n```shell\naws dynamodb create-table \\\n    --attribute-definitions \\\n        AttributeName=id,AttributeType=S \\\n    --key-schema "AttributeName=id,KeyType=HASH" \\\n    --provisioned-throughput "ReadCapacityUnits=5,WriteCapacityUnits=5" \\\n    --table-name app_session \n```\n\nSessions are intended to operate just like the default Flask session implementation:\n\n```python\nfrom flask import Flask, session\nfrom dynamodb_session_flask import DynamoDbSession\n\nflask_app = Flask(__name__)\nflask_app.session_interface = DynamoDbSession()\n\n@flask_app.route(\'/save\')\ndef save():\n    session[\'val\'] = \'My Value\'\n    return \'Success\', 200\n\n@flask_app.route(\'/load\')\ndef load():\n    saved_val = session[\'val\']\n    return saved_val, 200\n\n@flask_app.route(\'/end\')\ndef end_session():\n    # This will remove the session from the database and remove the session ID from cookies/headers\n    session.clear()\n    return \'Success\', 200\n```\n\n## Configuration\n\nThere are additional configuration options, and are set like normal Flask configuration:\n\n```python\nflask_app = Flask(__name__)\nflask_app.config.update(\n    SESSION_DYNAMODB_IDLE_TIMEOUT=600\n)\n```\n\nAll configuration is optional, assuming the defaults are okay.\n\n<dl>\n\n<dt><code>SESSION_DYNAMODB_ABSOLUTE_TIMEOUT</code></dt>\n<dd>\nAbsolute session timeout (in seconds).\n\nNote: This setting works in conjunction with Flask\'s [`PERMANENT_SESSION_LIFETIME`](https://flask.palletsprojects.com/en/2.0.x/config/#PERMANENT_SESSION_LIFETIME) setting.  The absolute timeout chosen will be whichever is less.\n\nDefault: `43200` (12 hours)\n</dd>\n\n<dt><code>SESSION_DYNAMODB_ENDPOINT_URL</code></dt>\n<dd>\nThe DynamoDB URL.\n\nDefault: `None` (i.e. Boto3 logic)\n</dd>\n\n<dt><code>SESSION_DYNAMODB_HEADER_NAME</code></dt>\n<dd>\nThe name of the header to use for the session ID.\n\nDefault: `x-id`\n</dd>\n\n<dt><code>SESSION_DYNAMODB_IDLE_TIMEOUT</code></dt>\n<dd>\nIdle session timeout (in seconds).\n\nDefault: `7200` (2 hours)\n</dd>\n\n<dt><code>SESSION_DYNAMODB_SID_BYTE_LENGTH</code></dt>\n<dd>\nSession ID length in bytes. \n\nThis does not correlate to the character length of the ID, which will be either:\n\n* 43 - How many characters a 32-byte value uses when Base64 encoded. \n* 71 - The 43 characters from the previous bullet, plus a dot and finally a 27-character HMAC signature. \n\nDefault: `32`\n</dd>\n\n<dt><code>SESSION_DYNAMODB_SID_KEYS</code></dt>\n<dd>\nFor a slightly more secure session ID, the key can be signed using a configurable and rotatable key. \n\nThe signature is generated using [`itsdangerous`](https://itsdangerous.palletsprojects.com/en/2.1.x/) and includes key rotation. If/When rotation is desired, the array is used in order from oldest to newest. Otherwise, one key is all that is needed.\n\nAn empty array means no signature is generated.\n\nDefault: `[]` (no signature)\n</dd>\n\n<dt><code>SESSION_DYNAMODB_TABLE_NAME</code></dt>\n<dd>\nThe name of the DynamoDB table.\n\nDefault: `app_session`\n</dd>\n\n<dt><code>SESSION_DYNAMODB_OVERRIDE_COOKIE_NAME</code></dt>\n<dd>\nWhether or not to override Flask\'s [SESSION_COOKIE_NAME](https://flask.palletsprojects.com/en/2.0.x/config/#SESSION_COOKIE_NAME)\nconfiguration for the session ID. While somewhat trivial, OWASP\'s recommended value is \n`id` and Flask\'s default is `session`. So to avoid using Flask\'s default or modifying it behind the scenes, this setting\nhelps separate this library\'s preferred default from Flask\'s.\n\nSetting this to `True` will set the cookie name to `id`. Otherwise, Flask\'s configuration will be used.\n\nDefault: `True`\n</dd>\n\n<dt><code>SESSION_DYNAMODB_OVERRIDE_COOKIE_SECURE</code></dt>\n<dd>\nWhether or not to override Flask\'s [`SESSION_COOKIE_SECURE`](https://flask.palletsprojects.com/en/2.0.x/config/#SESSION_COOKIE_SECURE)\nfor the cookie\'s Secure attribute. Flask defaults that attribute to `False`, whereas this should ideally be `True` to prevent \nMan-in-the-Middle attacks. \n\nSetting this to `True` will force the Secure attribute to also be `True`. Otherwise, Flask\'s configuration will be used.\n\nNote: You\'ll want to set this to `False` in any environment where TLS is not used (e.g. local development).\n\nDefault: `True`\n</dd>\n\n<dt><code>SESSION_DYNAMODB_USE_HEADER</code></dt>\n<dd>\nWhether or not to communicate/expect the session ID via headers.\n\nDefault: `False`\n</dd>\n\n<dt><code>SESSION_COOKIE_SAMESITE</code></dt>\n<dd>\nThis is actually a Flask configuration, which defaults to `None`. However, if the value is `None`, then we set it to \n`Strict` by default.\n\nDefault: `Strict` (indirectly changed)\n</dd>\n\n</dl>\n',
    'author': 'Jason Capriotti',
    'author_email': 'jason.capriotti@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/JCapriotti/dynamodb-session-flask',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
