"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuildProject = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const smithy_generated_output_1 = require("./components/smithy-generated-output");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyBuildProject extends projen_1.Project {
    constructor(options) {
        super(options);
        // Store whether we've synthesized the project
        this.synthed = false;
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        // Add the gradle files which the user may modify to customise what's generated
        new projen_1.SampleFile(this, "build.gradle", {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "smithy", "build.gradle"),
        });
        new projen_1.SampleFile(this, "settings.gradle", {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "smithy", "settings.gradle"),
        });
        // Create the smithy build json file
        new smithy_build_1.SmithyBuild(this, {
            version: "2.0",
            ...options.smithyBuildOptions,
            projections: {
                ...options.smithyBuildOptions?.projections,
                openapi: {
                    plugins: {
                        openapi: {
                            service: options.fullyQualifiedServiceName,
                            ...options.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
        });
        // SmithyBuild component above always writes to smithy-build.json
        this.smithyBuildConfigPath = path.join(this.outdir, "smithy-build.json");
        this.smithyBuildOutputPath = path.join(this.outdir, options.buildOutputDir);
        new smithy_generated_output_1.SmithyGeneratedOutput(this, {
            modelPath: options.modelPath,
            gradleProjectPath: this.outdir,
            smithyBuildConfigPath: this.smithyBuildConfigPath,
            outputPath: this.smithyBuildOutputPath,
            gradleWrapperPath: options.gradleWrapperPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.SmithyBuildProject = SmithyBuildProject;
//# sourceMappingURL=data:application/json;base64,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