"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupSmithyBuild = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const smithy_build_project_1 = require("./smithy-build-project");
const smithy_model_project_1 = require("./smithy-model-project");
/**
 * Adds the common subprojects for building a Smithy model into an OpenAPI spec
 */
exports.setupSmithyBuild = (project, options) => {
    const modelDir = options.modelDir ?? "model";
    const { namespace: serviceNamespace, serviceName } = options.serviceName;
    const fullyQualifiedServiceName = `${serviceNamespace}#${serviceName}`;
    const smithyBuildDir = "smithy-build";
    // Create a smithy model (eg main.smithy)
    const smithyModel = new smithy_model_project_1.SmithyModelProject({
        name: `${project.name}-smithy`,
        outdir: modelDir,
        parent: project,
        serviceNamespace,
        serviceName,
    });
    smithyModel.synth();
    const smithyBuildOutputSubDir = "output";
    // Create the smithy build project, responsible for transforming the model into an OpenAPI spec
    const smithyBuild = new smithy_build_project_1.SmithyBuildProject({
        name: `${project.name}-smithy-build`,
        parent: project,
        outdir: smithyBuildDir,
        fullyQualifiedServiceName,
        smithyBuildOptions: options.smithyBuildOptions,
        modelPath: path.join(project.outdir, modelDir),
        buildOutputDir: smithyBuildOutputSubDir,
        gradleWrapperPath: options.gradleWrapperPath
            ? path.resolve(project.outdir, options.gradleWrapperPath)
            : undefined,
    });
    smithyBuild.synth();
    const smithyBuildOutputDir = path.join(smithyBuildDir, smithyBuildOutputSubDir);
    // Ignore smithy build output by default
    if (options.ignoreSmithyBuildOutput ?? true) {
        project.gitignore.addPatterns(smithyBuildOutputDir);
    }
    // Ignore the .gradle directory
    project.gitignore.addPatterns(path.join(smithyBuildDir, ".gradle"));
    return {
        modelDir,
        // Generated spec is written to output/<projection id>/<plugin id>/<service name>.openapi.json
        generatedSpecFilePath: path.join(smithyBuild.smithyBuildOutputPath, "openapi", "openapi", `${serviceName}.openapi.json`),
    };
};
//# sourceMappingURL=data:application/json;base64,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