"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuildProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const smithy_generated_output_1 = require("./components/smithy-generated-output");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyBuildProject extends projen_1.Project {
    constructor(options) {
        super(options);
        // Store whether we've synthesized the project
        this.synthed = false;
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        // Add the gradle files which the user may modify to customise what's generated
        new projen_1.SampleFile(this, "build.gradle", {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "smithy", "build.gradle"),
        });
        new projen_1.SampleFile(this, "settings.gradle", {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "smithy", "settings.gradle"),
        });
        // Create the smithy build json file
        new smithy_build_1.SmithyBuild(this, {
            version: "2.0",
            ...options.smithyBuildOptions,
            projections: {
                ...options.smithyBuildOptions?.projections,
                openapi: {
                    plugins: {
                        openapi: {
                            service: options.fullyQualifiedServiceName,
                            // By default, preserve tags in the generated spec, but allow users to explicitly overwrite this
                            tags: true,
                            ...options.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
        });
        // SmithyBuild component above always writes to smithy-build.json
        this.smithyBuildConfigPath = path.join(this.outdir, "smithy-build.json");
        this.smithyBuildOutputPath = path.join(this.outdir, options.buildOutputDir);
        new smithy_generated_output_1.SmithyGeneratedOutput(this, {
            modelPath: options.modelPath,
            gradleProjectPath: this.outdir,
            smithyBuildConfigPath: this.smithyBuildConfigPath,
            outputPath: this.smithyBuildOutputPath,
            gradleWrapperPath: options.gradleWrapperPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.SmithyBuildProject = SmithyBuildProject;
//# sourceMappingURL=data:application/json;base64,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