"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypescriptSampleTests = exports.getTypescriptSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample typescript source code
 */
exports.getTypescriptSampleSource = (options) => ({
    "index.ts": [
        "export * from './api';",
        ...(options.sampleCode !== false ? ["export * from './sample-api';"] : []),
    ].join("\n"),
    // This file provides a type-safe interface to the exported OpenApiGatewayRestApi construct
    "api.ts": `import { OpenApiGatewayRestApi, OpenApiGatewayRestApiProps, OpenApiIntegration } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { OperationLookup, OperationConfig } from "${options.typescriptClientPackageName}";
import * as path from "path";
import spec from "../${options.specDir}/${options.parsedSpecFileName}";

export type ApiIntegrations = OperationConfig<OpenApiIntegration>;

export interface ApiProps extends Omit<OpenApiGatewayRestApiProps, "spec" | "specPath" | "operationLookup" | "integrations"> {
  readonly integrations: ApiIntegrations;
}

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
export class Api extends OpenApiGatewayRestApi {
  constructor(scope: Construct, id: string, props: ApiProps) {
    super(scope, id, {
      ...props,
      integrations: props.integrations as any,
      spec,
      specPath: path.resolve(__dirname, "../${options.specDir}/${options.parsedSpecFileName}"),
      operationLookup: OperationLookup as any,
    });
  }
}
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample-api.ts": `import { Authorizers, Integrations } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { Cors } from "aws-cdk-lib/aws-apigateway";
import { NodejsFunction } from "aws-cdk-lib/aws-lambda-nodejs";
import { Api } from "./api";

/**
 * An example of how to wire lambda handler functions to the API
 */
export class SampleApi extends Api {
  constructor(scope: Construct, id: string) {
    super(scope, id, {
      defaultAuthorizer: Authorizers.iam(),
      corsOptions: {
        allowOrigins: Cors.ALL_ORIGINS,
        allowMethods: Cors.ALL_METHODS,
      },
      integrations: {
        sayHello: {
          integration: Integrations.lambda(new NodejsFunction(scope, "say-hello")),
        },
      },
    });
  }
}
`,
            // Generate an example lambda handler
            "sample-api.say-hello.ts": `import { sayHelloHandler } from "${options.typescriptClientPackageName}";

/**
 * An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs.
 */
export const handler = sayHelloHandler(async ({ input }) => {
  return {
    statusCode: 200,
    headers: {
      "Access-Control-Allow-Origin": "*",
      "Access-Control-Allow-Headers": "*",
    },
    body: {
      message: \`Hello \${input.requestParameters.name}!\`,
    },
  };
});
`,
        }
        : {}),
});
/**
 * Return a map of file name to contents for the sample typescript test code
 */
exports.getTypescriptSampleTests = (options) => ({
    "api.test.ts": `import { Integrations } from "${options.openApiGatewayPackageName}";
import { Stack } from "aws-cdk-lib";
import { Template } from "aws-cdk-lib/assertions";
import { Code, Function, Runtime } from "aws-cdk-lib/aws-lambda";
import { OperationLookup } from "${options.typescriptClientPackageName}";
import { Api } from "../${options.apiSrcDir}";

/**
 * A simple test to ensure the api construct synthesizes correctly
 */
describe("Api", () => {
  it("should synthesize", () => {
    const stack = new Stack();
    new Api(stack, "ApiTest", {
      // Create a dummy integration for every operation defined in the api
      integrations: Object.fromEntries(Object.keys(OperationLookup).map((operation) => [operation, {
        integration: Integrations.lambda(new Function(stack, \`\${operation}Lambda\`, {
          code: Code.fromInline("test"), handler: "test", runtime: Runtime.NODEJS_14_X,
        })),
      }])) as any,
    });
  
    const template = Template.fromStack(stack);
    expect(template.toJSON()).toMatchSnapshot();
  });
});
`,
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHlwZXNjcmlwdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9wcm9qZWN0L3NhbXBsZXMvdHlwZXNjcmlwdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFnQ0E7O0dBRUc7QUFDVSxRQUFBLHlCQUF5QixHQUFHLENBQ3ZDLE9BQW9DLEVBQ0osRUFBRSxDQUFDLENBQUM7SUFDcEMsVUFBVSxFQUFFO1FBQ1Ysd0JBQXdCO1FBQ3hCLEdBQUcsQ0FBQyxPQUFPLENBQUMsVUFBVSxLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7S0FDM0UsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO0lBQ1osMkZBQTJGO0lBQzNGLFFBQVEsRUFBRSwwRkFBMEYsT0FBTyxDQUFDLHlCQUF5Qjs7b0RBRW5GLE9BQU8sQ0FBQywyQkFBMkI7O3VCQUVoRSxPQUFPLENBQUMsT0FBTyxJQUFJLE9BQU8sQ0FBQyxrQkFBa0I7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs4Q0FrQnRCLE9BQU8sQ0FBQyxPQUFPLElBQUksT0FBTyxDQUFDLGtCQUFrQjs7Ozs7Q0FLMUY7SUFDQyxHQUFHLENBQUMsT0FBTyxDQUFDLFVBQVUsS0FBSyxLQUFLO1FBQzlCLENBQUMsQ0FBQztZQUNFLDJEQUEyRDtZQUMzRCw2REFBNkQ7WUFDN0QsZUFBZSxFQUFFLDhDQUE4QyxPQUFPLENBQUMseUJBQXlCOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0NBeUJ2RztZQUNPLHFDQUFxQztZQUNyQyx5QkFBeUIsRUFBRSxvQ0FBb0MsT0FBTyxDQUFDLDJCQUEyQjs7Ozs7Ozs7Ozs7Ozs7Ozs7Q0FpQnpHO1NBQ007UUFDSCxDQUFDLENBQUMsRUFBRSxDQUFDO0NBQ1IsQ0FBQyxDQUFDO0FBRUg7O0dBRUc7QUFDVSxRQUFBLHdCQUF3QixHQUFHLENBQ3RDLE9BQW9DLEVBQ3BDLEVBQUUsQ0FBQyxDQUFDO0lBQ0osYUFBYSxFQUFFLGlDQUFpQyxPQUFPLENBQUMseUJBQXlCOzs7O21DQUloRCxPQUFPLENBQUMsMkJBQTJCOzBCQUM1QyxPQUFPLENBQUMsU0FBUzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0NBcUIxQztDQUNBLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuLyoqXG4gKiBPcHRpb25zIGZvciB0eXBlc2NyaXB0IHNhbXBsZSBjb2RlIGdlbmVyYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUeXBlc2NyaXB0U2FtcGxlQ29kZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHBhY2thZ2UgbmFtZSB3aGljaCBleHBvcnRzIHRoZSBPcGVuQXBpR2F0ZXdheUxhbWJkYUFwaSBjb25zdHJ1Y3QgKGllIHRoaXMgcGRrIHBhY2thZ2UhKVxuICAgKi9cbiAgcmVhZG9ubHkgb3BlbkFwaUdhdGV3YXlQYWNrYWdlTmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGdlbmVyYXRlZCB0eXBlc2NyaXB0IGNsaWVudCBwYWNrYWdlXG4gICAqL1xuICByZWFkb25seSB0eXBlc2NyaXB0Q2xpZW50UGFja2FnZU5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRvIGdlbmVyYXRlIHNhbXBsZSBjb2RlXG4gICAqL1xuICByZWFkb25seSBzYW1wbGVDb2RlPzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIEFwaSBzb3VyY2UgY29kZSBkaXJlY3RvcnksIHJlbGF0aXZlIHRvIHRoZSBwcm9qZWN0IHJvb3RcbiAgICovXG4gIHJlYWRvbmx5IGFwaVNyY0Rpcjogc3RyaW5nO1xuICAvKipcbiAgICogRGlyZWN0b3J5IHdoZXJlIHRoZSBwYXJzZWQgc3BlYyBpcyBvdXRwdXRcbiAgICovXG4gIHJlYWRvbmx5IHNwZWNEaXI6IHN0cmluZztcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIHBhcnNlZCBzcGVjIGZpbGVcbiAgICovXG4gIHJlYWRvbmx5IHBhcnNlZFNwZWNGaWxlTmFtZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIFJldHVybiBhIG1hcCBvZiBmaWxlIG5hbWUgdG8gY29udGVudHMgZm9yIHRoZSBzYW1wbGUgdHlwZXNjcmlwdCBzb3VyY2UgY29kZVxuICovXG5leHBvcnQgY29uc3QgZ2V0VHlwZXNjcmlwdFNhbXBsZVNvdXJjZSA9IChcbiAgb3B0aW9uczogVHlwZXNjcmlwdFNhbXBsZUNvZGVPcHRpb25zXG4pOiB7IFtmaWxlTmFtZTogc3RyaW5nXTogc3RyaW5nIH0gPT4gKHtcbiAgXCJpbmRleC50c1wiOiBbXG4gICAgXCJleHBvcnQgKiBmcm9tICcuL2FwaSc7XCIsXG4gICAgLi4uKG9wdGlvbnMuc2FtcGxlQ29kZSAhPT0gZmFsc2UgPyBbXCJleHBvcnQgKiBmcm9tICcuL3NhbXBsZS1hcGknO1wiXSA6IFtdKSxcbiAgXS5qb2luKFwiXFxuXCIpLFxuICAvLyBUaGlzIGZpbGUgcHJvdmlkZXMgYSB0eXBlLXNhZmUgaW50ZXJmYWNlIHRvIHRoZSBleHBvcnRlZCBPcGVuQXBpR2F0ZXdheVJlc3RBcGkgY29uc3RydWN0XG4gIFwiYXBpLnRzXCI6IGBpbXBvcnQgeyBPcGVuQXBpR2F0ZXdheVJlc3RBcGksIE9wZW5BcGlHYXRld2F5UmVzdEFwaVByb3BzLCBPcGVuQXBpSW50ZWdyYXRpb24gfSBmcm9tIFwiJHtvcHRpb25zLm9wZW5BcGlHYXRld2F5UGFja2FnZU5hbWV9XCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgT3BlcmF0aW9uTG9va3VwLCBPcGVyYXRpb25Db25maWcgfSBmcm9tIFwiJHtvcHRpb25zLnR5cGVzY3JpcHRDbGllbnRQYWNrYWdlTmFtZX1cIjtcbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCBzcGVjIGZyb20gXCIuLi8ke29wdGlvbnMuc3BlY0Rpcn0vJHtvcHRpb25zLnBhcnNlZFNwZWNGaWxlTmFtZX1cIjtcblxuZXhwb3J0IHR5cGUgQXBpSW50ZWdyYXRpb25zID0gT3BlcmF0aW9uQ29uZmlnPE9wZW5BcGlJbnRlZ3JhdGlvbj47XG5cbmV4cG9ydCBpbnRlcmZhY2UgQXBpUHJvcHMgZXh0ZW5kcyBPbWl0PE9wZW5BcGlHYXRld2F5UmVzdEFwaVByb3BzLCBcInNwZWNcIiB8IFwic3BlY1BhdGhcIiB8IFwib3BlcmF0aW9uTG9va3VwXCIgfCBcImludGVncmF0aW9uc1wiPiB7XG4gIHJlYWRvbmx5IGludGVncmF0aW9uczogQXBpSW50ZWdyYXRpb25zO1xufVxuXG4vKipcbiAqIFR5cGUtc2FmZSBjb25zdHJ1Y3QgZm9yIHRoZSBBUEkgR2F0ZXdheSByZXNvdXJjZXMgZGVmaW5lZCBieSB0aGUgc3BlYy5cbiAqIFlvdSB3aWxsIGxpa2VseSBub3QgbmVlZCB0byBtb2RpZnkgdGhpcyBmaWxlLCBhbmQgY2FuIGluc3RlYWQgZXh0ZW5kIGl0IGFuZCBkZWZpbmUgeW91ciBpbnRlZ3JhdGlvbnMuXG4gKi9cbmV4cG9ydCBjbGFzcyBBcGkgZXh0ZW5kcyBPcGVuQXBpR2F0ZXdheVJlc3RBcGkge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBpUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIC4uLnByb3BzLFxuICAgICAgaW50ZWdyYXRpb25zOiBwcm9wcy5pbnRlZ3JhdGlvbnMgYXMgYW55LFxuICAgICAgc3BlYyxcbiAgICAgIHNwZWNQYXRoOiBwYXRoLnJlc29sdmUoX19kaXJuYW1lLCBcIi4uLyR7b3B0aW9ucy5zcGVjRGlyfS8ke29wdGlvbnMucGFyc2VkU3BlY0ZpbGVOYW1lfVwiKSxcbiAgICAgIG9wZXJhdGlvbkxvb2t1cDogT3BlcmF0aW9uTG9va3VwIGFzIGFueSxcbiAgICB9KTtcbiAgfVxufVxuYCxcbiAgLi4uKG9wdGlvbnMuc2FtcGxlQ29kZSAhPT0gZmFsc2VcbiAgICA/IHtcbiAgICAgICAgLy8gR2VuZXJhdGUgYW4gZXhhbXBsZSB3aGljaCBpbnN0YW50aWF0ZXMgdGhlIEFwaSBjb25zdHJ1Y3RcbiAgICAgICAgLy8gVE9ETzogQ29uc2lkZXIgZ2VuZXJhdGluZyB0aGlzIHNhbXBsZSBmcm9tIHRoZSBwYXJzZWQgc3BlY1xuICAgICAgICBcInNhbXBsZS1hcGkudHNcIjogYGltcG9ydCB7IEF1dGhvcml6ZXJzLCBJbnRlZ3JhdGlvbnMgfSBmcm9tIFwiJHtvcHRpb25zLm9wZW5BcGlHYXRld2F5UGFja2FnZU5hbWV9XCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHsgQ29ycyB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtYXBpZ2F0ZXdheVwiO1xuaW1wb3J0IHsgTm9kZWpzRnVuY3Rpb24gfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxhbWJkYS1ub2RlanNcIjtcbmltcG9ydCB7IEFwaSB9IGZyb20gXCIuL2FwaVwiO1xuXG4vKipcbiAqIEFuIGV4YW1wbGUgb2YgaG93IHRvIHdpcmUgbGFtYmRhIGhhbmRsZXIgZnVuY3Rpb25zIHRvIHRoZSBBUElcbiAqL1xuZXhwb3J0IGNsYXNzIFNhbXBsZUFwaSBleHRlbmRzIEFwaSB7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIGRlZmF1bHRBdXRob3JpemVyOiBBdXRob3JpemVycy5pYW0oKSxcbiAgICAgIGNvcnNPcHRpb25zOiB7XG4gICAgICAgIGFsbG93T3JpZ2luczogQ29ycy5BTExfT1JJR0lOUyxcbiAgICAgICAgYWxsb3dNZXRob2RzOiBDb3JzLkFMTF9NRVRIT0RTLFxuICAgICAgfSxcbiAgICAgIGludGVncmF0aW9uczoge1xuICAgICAgICBzYXlIZWxsbzoge1xuICAgICAgICAgIGludGVncmF0aW9uOiBJbnRlZ3JhdGlvbnMubGFtYmRhKG5ldyBOb2RlanNGdW5jdGlvbihzY29wZSwgXCJzYXktaGVsbG9cIikpLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxufVxuYCxcbiAgICAgICAgLy8gR2VuZXJhdGUgYW4gZXhhbXBsZSBsYW1iZGEgaGFuZGxlclxuICAgICAgICBcInNhbXBsZS1hcGkuc2F5LWhlbGxvLnRzXCI6IGBpbXBvcnQgeyBzYXlIZWxsb0hhbmRsZXIgfSBmcm9tIFwiJHtvcHRpb25zLnR5cGVzY3JpcHRDbGllbnRQYWNrYWdlTmFtZX1cIjtcblxuLyoqXG4gKiBBbiBleGFtcGxlIGxhbWJkYSBoYW5kbGVyIHdoaWNoIHVzZXMgdGhlIGdlbmVyYXRlZCBoYW5kbGVyIHdyYXBwZXIgdG8gbWFuYWdlIG1hcnNoYWxsaW5nIGlucHV0cy9vdXRwdXRzLlxuICovXG5leHBvcnQgY29uc3QgaGFuZGxlciA9IHNheUhlbGxvSGFuZGxlcihhc3luYyAoeyBpbnB1dCB9KSA9PiB7XG4gIHJldHVybiB7XG4gICAgc3RhdHVzQ29kZTogMjAwLFxuICAgIGhlYWRlcnM6IHtcbiAgICAgIFwiQWNjZXNzLUNvbnRyb2wtQWxsb3ctT3JpZ2luXCI6IFwiKlwiLFxuICAgICAgXCJBY2Nlc3MtQ29udHJvbC1BbGxvdy1IZWFkZXJzXCI6IFwiKlwiLFxuICAgIH0sXG4gICAgYm9keToge1xuICAgICAgbWVzc2FnZTogXFxgSGVsbG8gXFwke2lucHV0LnJlcXVlc3RQYXJhbWV0ZXJzLm5hbWV9IVxcYCxcbiAgICB9LFxuICB9O1xufSk7XG5gLFxuICAgICAgfVxuICAgIDoge30pLFxufSk7XG5cbi8qKlxuICogUmV0dXJuIGEgbWFwIG9mIGZpbGUgbmFtZSB0byBjb250ZW50cyBmb3IgdGhlIHNhbXBsZSB0eXBlc2NyaXB0IHRlc3QgY29kZVxuICovXG5leHBvcnQgY29uc3QgZ2V0VHlwZXNjcmlwdFNhbXBsZVRlc3RzID0gKFxuICBvcHRpb25zOiBUeXBlc2NyaXB0U2FtcGxlQ29kZU9wdGlvbnNcbikgPT4gKHtcbiAgXCJhcGkudGVzdC50c1wiOiBgaW1wb3J0IHsgSW50ZWdyYXRpb25zIH0gZnJvbSBcIiR7b3B0aW9ucy5vcGVuQXBpR2F0ZXdheVBhY2thZ2VOYW1lfVwiO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IFRlbXBsYXRlIH0gZnJvbSBcImF3cy1jZGstbGliL2Fzc2VydGlvbnNcIjtcbmltcG9ydCB7IENvZGUsIEZ1bmN0aW9uLCBSdW50aW1lIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1sYW1iZGFcIjtcbmltcG9ydCB7IE9wZXJhdGlvbkxvb2t1cCB9IGZyb20gXCIke29wdGlvbnMudHlwZXNjcmlwdENsaWVudFBhY2thZ2VOYW1lfVwiO1xuaW1wb3J0IHsgQXBpIH0gZnJvbSBcIi4uLyR7b3B0aW9ucy5hcGlTcmNEaXJ9XCI7XG5cbi8qKlxuICogQSBzaW1wbGUgdGVzdCB0byBlbnN1cmUgdGhlIGFwaSBjb25zdHJ1Y3Qgc3ludGhlc2l6ZXMgY29ycmVjdGx5XG4gKi9cbmRlc2NyaWJlKFwiQXBpXCIsICgpID0+IHtcbiAgaXQoXCJzaG91bGQgc3ludGhlc2l6ZVwiLCAoKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBuZXcgQXBpKHN0YWNrLCBcIkFwaVRlc3RcIiwge1xuICAgICAgLy8gQ3JlYXRlIGEgZHVtbXkgaW50ZWdyYXRpb24gZm9yIGV2ZXJ5IG9wZXJhdGlvbiBkZWZpbmVkIGluIHRoZSBhcGlcbiAgICAgIGludGVncmF0aW9uczogT2JqZWN0LmZyb21FbnRyaWVzKE9iamVjdC5rZXlzKE9wZXJhdGlvbkxvb2t1cCkubWFwKChvcGVyYXRpb24pID0+IFtvcGVyYXRpb24sIHtcbiAgICAgICAgaW50ZWdyYXRpb246IEludGVncmF0aW9ucy5sYW1iZGEobmV3IEZ1bmN0aW9uKHN0YWNrLCBcXGBcXCR7b3BlcmF0aW9ufUxhbWJkYVxcYCwge1xuICAgICAgICAgIGNvZGU6IENvZGUuZnJvbUlubGluZShcInRlc3RcIiksIGhhbmRsZXI6IFwidGVzdFwiLCBydW50aW1lOiBSdW50aW1lLk5PREVKU18xNF9YLFxuICAgICAgICB9KSksXG4gICAgICB9XSkpIGFzIGFueSxcbiAgICB9KTtcbiAgXG4gICAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICAgIGV4cGVjdCh0ZW1wbGF0ZS50b0pTT04oKSkudG9NYXRjaFNuYXBzaG90KCk7XG4gIH0pO1xufSk7XG5gLFxufSk7XG4iXX0=