"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientSettings = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
// initialize logger
const logger = log4js_1.getLogger();
/**
 * Component for parsing the yaml OpenAPI spec as a single json object, resolving references etc.
 */
class ClientSettings extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.options = options;
        this.clientSettingsFilename =
            options.clientSettingsFilename ?? ".client-settings.json";
        this.clientSettingsPath = path.join(options.generatedCodeDir, this.clientSettingsFilename);
        // load previously generated client config
        const clientSettingsPathAbs = path.join(project.outdir, this.clientSettingsPath);
        logger.trace(`Reading client settings from ${clientSettingsPathAbs}`);
        const existingClientConfig = util_1.tryReadFileSync(clientSettingsPathAbs);
        const prevClientLanguages = new Set();
        const prevDocFormats = new Set();
        if (existingClientConfig) {
            const parsedClientConfig = JSON.parse(existingClientConfig);
            // previously generated client settings
            parsedClientConfig.clientLanguages.map((l) => prevClientLanguages.add(l));
            parsedClientConfig.documentationFormats.map((d) => prevDocFormats.add(d));
        }
        this.clientLanguageConfigs = options.clientLanguages.map((clientLanguage) => ({
            clientLanguage,
            isDefault: clientLanguage === options.defaultClientLanguage,
            generate: options.specChanged ||
                options.forceGenerateCodeAndDocs ||
                !prevClientLanguages.has(clientLanguage),
        }));
        this.documentationFormatConfigs = options.documentationFormats.map((documentationFormat) => ({
            documentationFormat,
            generate: options.specChanged ||
                options.forceGenerateCodeAndDocs ||
                !prevDocFormats.has(documentationFormat),
        }));
    }
    synthesize() {
        // store the client config
        // this helps to optimize to re-generate projects when new language/doc format is
        // introduced while the spec doesn't change
        logger.trace(`Generating settings file to ${this.clientSettingsPath}`);
        const clientSettingsJsonFile = new projen_1.JsonFile(this.project, this.clientSettingsPath, {
            obj: {
                clientLanguages: this.options.clientLanguages,
                documentationFormats: this.options.documentationFormats,
            },
            marker: true,
            editGitignore: false,
        });
        logger.debug(`Generated settings file to ${clientSettingsJsonFile.absolutePath}`);
    }
}
exports.ClientSettings = ClientSettings;
//# sourceMappingURL=data:application/json;base64,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