"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appsync_1 = require("aws-cdk-lib/aws-appsync");
const private_1 = require("./private");
const schema_intermediate_1 = require("./schema-intermediate");
;
/**
 * (experimental) The Schema for a GraphQL Api.
 *
 * If no options are configured, schema will be generated
 * code-first.
 *
 * @experimental
 */
class Schema {
    /**
     * @experimental
     */
    constructor(options) {
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
        this.types = [];
    }
    /**
     * (experimental) Generate a Schema from file.
     *
     * @param filePath the file path of the schema file.
     * @returns `SchemaAsset` with immutable schema defintion
     * @experimental
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * (experimental) Called when the GraphQL Api is initialized to allow this object to bind to the stack.
     *
     * @param api The binding GraphQL Api.
     * @experimental
     */
    bind(api) {
        if (!this.schema) {
            this.schema = new aws_appsync_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: this.mode === private_1.SchemaMode.CODE ?
                    aws_cdk_lib_1.Lazy.string({
                        produce: () => this.types.reduce((acc, type) => `${acc}${type._bindToGraphqlApi(api).toString()}\n`, `${this.declareSchema()}${this.definition}`),
                    })
                    : this.definition,
            });
        }
        return this.schema;
    }
    /**
     * (experimental) Escape hatch to add to Schema as desired.
     *
     * Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema.
     * @param delimiter the delimiter between schema and addition.
     * @default - ''
     * @experimental
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * (experimental) Add a query field to the schema's Query. CDK will create an Object Type called 'Query'. For example,.
     *
     * type Query {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query.
     * @param field the resolvable field to for this query.
     * @experimental
     */
    addQuery(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add query. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.query) {
            this.query = new schema_intermediate_1.ObjectType('Query', { definition: {} });
            this.addType(this.query);
        }
        ;
        this.query.addField({ fieldName, field });
        return this.query;
    }
    /**
     * (experimental) Add a mutation field to the schema's Mutation. CDK will create an Object Type called 'Mutation'. For example,.
     *
     * type Mutation {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation.
     * @param field the resolvable field to for this Mutation.
     * @experimental
     */
    addMutation(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add mutation. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.mutation) {
            this.mutation = new schema_intermediate_1.ObjectType('Mutation', { definition: {} });
            this.addType(this.mutation);
        }
        ;
        this.mutation.addField({ fieldName, field });
        return this.mutation;
    }
    /**
     * (experimental) Add a subscription field to the schema's Subscription. CDK will create an Object Type called 'Subscription'. For example,.
     *
     * type Subscription {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription.
     * @param field the resolvable field to for this Subscription.
     * @experimental
     */
    addSubscription(fieldName, field) {
        var _b, _c;
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add subscription. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.subscription) {
            this.subscription = new schema_intermediate_1.ObjectType('Subscription', { definition: {} });
            this.addType(this.subscription);
        }
        const directives = (_c = (_b = field.fieldOptions) === null || _b === void 0 ? void 0 : _b.directives) === null || _c === void 0 ? void 0 : _c.filter((directive) => directive.mutationFields);
        if (directives && directives.length > 1) {
            throw new Error(`Subscription fields must not have more than one @aws_subscribe directives. Received: ${directives.length}`);
        }
        this.subscription.addField({ fieldName, field });
        return this.subscription;
    }
    /**
     * (experimental) Add type to the schema.
     *
     * @param type the intermediate type to add to the schema.
     * @experimental
     */
    addType(type) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.types.push(type);
        return type;
    }
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    declareSchema() {
        if (!this.query && !this.mutation && !this.subscription) {
            return '';
        }
        const list = ['query', 'mutation', 'subscription'];
        return private_1.shapeAddition({
            prefix: 'schema',
            fields: list.map((key) => { var _b; return this[key] ? `${key}: ${(_b = this[key]) === null || _b === void 0 ? void 0 : _b.name}` : ''; })
                .filter((field) => field != ''),
        }) + '\n';
    }
}
exports.Schema = Schema;
_a = JSII_RTTI_SYMBOL_1;
Schema[_a] = { fqn: "@aws-cdk/aws-appsync-alpha.Schema", version: "2.0.0-alpha.6" };
//# sourceMappingURL=data:application/json;base64,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