# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['asynctinydb']

package_data = \
{'': ['*']}

install_requires = \
['aiofiles>=22.1.0,<23.0.0',
 'nest-asyncio>=1.5.5,<2.0.0',
 'ujson>=5.4.0,<6.0.0']

extras_require = \
{':python_version <= "3.10"': ['typing-extensions>=3.10.0,<5.0.0']}

setup_kwargs = {
    'name': 'async-tinydb',
    'version': '1.2.2',
    'description': 'Yet Another Async TinyDB',
    'long_description': '![logo](https://raw.githubusercontent.com/msiemens/tinydb/master/artwork/logo.png)\n\n## What\'s This?\n\n"An asynchronous IO version of `TinyDB` based on `aiofiles`."\n\nAlmost every method is asynchronous. And it\'s based on `TinyDB 4.7.0+`.  \nI will try to keep up with the latest version of `TinyDB`.\n\n## Major Changes\n* **asynchronous**: Say goodbye to blocking IO.\n  \n* **drop support**: Only supports Python 3.8+.\n  \n* **event hooks**: You can now use event hooks to do something before or after an operation. See [Event Hooks](#event-hooks) for more details.\n  \n* **redesigned id & doc class**: The ID and Document classes are more abstract. You can customise them more pleasingly.\n  The default ID class is `IncreID`, which mimics the behaviours of the original `int` ID but requires much fewer IO operations.\n\n* **db level caching**: This significantly improves the performance of all operations. But it requires more memory, and the responsibility of converting the data to the correct type is moved to the Storage. e.g. `JSONStorage` needs to convert the keys to `str` by itself.\n\n## Minor Changes:\n\n* **lazy-load:** When `access_mode` is set to `\'r\'`, `FileNotExistsError` is not raised until the first read operation.\n\n* **ujson:** Using `ujson` instead of `json`. Some arguments aren\'t compatible with `json`\n  Why not `orjson`? Because `ujson` is fast enough and has more features.\n  \n* **Storage `closed` property**: Original `TinyDB` won\'t raise exceptions when operating on a closed file. Now the property `closed` of `Storage` classes is required to be implemented. An `IOError should be raised.\n  \n  I strongly suggest doing the same for `middleware`.\n\n## How to use it?\n\n#### Installation\n\n```Bash\npip install async-tinydb\n```\n\n#### Importing\n```Python\nfrom asynctinydb import TinyDB, where\n```\n\n\nAll you need to do is insert an `await` before every method that needs IO.\n\nNotice that some parts of the code are blocking, for example, when calling `len()` on `TinyDB` or `Table` Objects.\n\n#### Event Hooks\nEvent Hooks give you more flexibility than middleware.\nFor example, you can achieve compress/decompress data without creating a new Storage class.\n\nCurrently only supports storage events: `write.pre`, `write.post`, `read.pre`, `read.post`, `close`.\n\n* `write.pre` is called before json dumping, args: `str`(event name), `Storage`, `dict`(data).\n\n* `write.post` is called after json dumping, args: `str`(event name), `Storage`, `str|bytes`(json str or bytes).\n  Only one function can be registered for this event. Return non `None` value will be written to the file.\n\n* `read.pre` is called before json loading, args: `str`(event name), `Storage`, `str|bytes`(json str or bytes).\n  Only one function can be registered for this event. Return non `None` value will be used as the data.\n\n* `read.post` is called after json loading, args: `str`(event name), `Storage`, `dict`(data).\n\n* `close` is called when the storage is closed, args: `str`(event name), `Storage`.\n\nFor `write.pre` and `read.post`, you can directly modify data to edit its content.\n\nHowever, `write.post` and `read.pre` requires you to return the value to update content because `str` is immutable in Python. If there is no return value or returns a `None`, you won\'t change anything.\n\n```Python\ns = Storage()\n# By accessing the attribute `on`, you can register a new func to the event\n@s.on.write.pre\nasync def f(ev, s, data):  # Will be executed on event `write.pre`\n  ...\n```\n\n\n\n## Example Codes:\n\n### Simple One\n\n```Python\nimport asyncio\nfrom asynctinydb import TinyDB, Query\n\nasync def main():\n    db = TinyDB(\'test.json\')\n    await db.insert({"answer": 42})\n    print(await db.search(Query().answer == 42))  # >>> [{\'answer\': 42}] \n\nasyncio.run(main())\n```\n### Event Hooks Example\n\n```Python\nasync def main():\n    db = TinyDB(\'test.json\')\n\n    @db.storage.on.write.pre\n    async def mul(ev: str, s: Storage, data: dict):\n        data["_default"]["1"][\'answer\'] *= 2  # directly manipulate on data\n\n    @db.storage.on.write.post\n    async def _print(ev, s, anystr):\n      \tprint(anystr)  # print json dumped string\n \n    await db.insert({"answer": 21})  # insert() will trigger both write events\n    await db.close()\n    # Reload\n    db = TinyDB(\'test.json\')\n    print(await db.search(Query().answer == 42))  # >>> [{\'answer\': 42}] \n```\n\n### Customise ID Class\n\nInherit from `BaseID` and implement the following methods, then you are good to go.\n\n```Python\nfrom asynctinydb import BaseID\n\nclass MyID(BaseID):\n  def __init__(self, value: Any):\n        """\n        Should be able to convert str into MyID instance if you want to use JSONStorage.\n        """\n\n    def __str__(self) -> str:\n        """\n        Optional.\n        Should be implement if you want to use JSONStorage.\n        """\n\n    def __hash__(self) -> int:\n        ...\n\n    def __eq__(self, other: object) -> bool:\n        ...\n\n    @classmethod\n    def next_id(cls, table: Table) -> IncreID:\n        """\n        Recommended to define as an async function, but a sync def will do.\n        Should return a unique ID.\n        """\n\n    @classmethod\n    def mark_existed(cls, table: Table, new_id: IncreID):\n        """\n        Marks an ID as existed, the same ID shouldn\'t be generated by next_id again.\n        """\n\n    @classmethod\n    def clear_cache(cls, table: Table):\n        """\n        Clear cache of existing IDs, if such cache exists.\n        """\n```\n\n### Customise Document Class\n\n```Python\nfrom asynctinydb import BaseDocument\n\nclass MyDoc(BaseDocument):\n  """\n  I am too lazy to write those necessary methods.\n  """\n```\n\nAnyways, a BaseDocument class looks like this:\n\n```Python\nclass BaseDocument(Mapping[IDVar, Any]):\n    @property\n    @abstractmethod\n    def doc_id(self) -> IDVar:\n        raise NotImplementedError()\n\n    @doc_id.setter\n    def doc_id(self, value: IDVar):\n        raise NotImplementedError()\n```\n\nMake sure you have implemented all the methods required by `Mapping` and `BaseDocument` classes.\n',
    'author': 'VermiIIi0n',
    'author_email': 'dungeon.behind0t@icloud.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/VermiIIi0n/async-tinydb',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
