"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3CrossAccount = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * This CDK construct grants cross account permissions on an Amazon S3 location.
 * It uses a bucket policy and an Amazon KMS Key policy if the bucket is encrypted with KMS.
 * The cross account permission is granted to the entire account and not to a specific principal in this account.
 * It's the responsibility of the target account to grant permissions to the relevant principals.
 *
 * Note that it uses a Bucket object and not an IBucket because CDK can only add policies to objects managed in the CDK stack.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { S3CrossAccount } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'S3CrossAccountStack');
 *
 * const myBucket = new Bucket(stack, 'MyBucket')
 *
 * new S3CrossAccount(stack, 'S3CrossAccountGrant', {
 *   bucket: myBucket,
 *   s3ObjectKey: 'my-data',
 *   accountId: '1234567891011',
 * });
 * ```
 */
class S3CrossAccount extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Get the target account as a Principal
        const targetAccount = new aws_iam_1.AccountPrincipal(props.accountId);
        // // Get the bucket from the S3 location to attache a bucket policy
        // props.bucket.addToResourcePolicy(
        //   new PolicyStatement({
        //     principals: [
        //       targetAccount,
        //     ],
        //     actions: [
        //       's3:GetObject',
        //       's3:PutObject',
        //       's3:DeleteObject',
        //       's3:ListBucketMultipartUploads',
        //       's3:ListMultipartUploadParts',
        //       's3:AbortMultipartUpload',
        //       's3:ListBucket',
        //     ],
        //     resources: [
        //       props.bucket.arnForObjects(props.objectKey || '') + '/*',
        //       props.bucket.bucketArn,
        //     ],
        //   }),
        // );
        // // If the bucket is encrypted with a custom KMS key, attach a policy to the key to grant encrypt and decrypt
        // if (props.bucket.encryptionKey)  props.bucket.encryptionKey.grantEncryptDecrypt(targetAccount);
        if (props.s3ObjectKey === undefined)
            props.s3Bucket.grantReadWrite(targetAccount);
        else
            props.s3Bucket.grantReadWrite(targetAccount, props.s3ObjectKey);
        props.s3Bucket.grantReadWrite(targetAccount, props.s3ObjectKey + '/*');
        // Grant target account access to encryption configuration of an S3 bucket
        props.s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            principals: [targetAccount],
            actions: [
                's3:GetEncryptionConfiguration'
            ],
            resources: [
                props.s3Bucket.bucketArn,
            ]
        }));
    }
}
exports.S3CrossAccount = S3CrossAccount;
_a = JSII_RTTI_SYMBOL_1;
S3CrossAccount[_a] = { fqn: "aws-analytics-reference-architecture.S3CrossAccount", version: "2.8.1" };
//# sourceMappingURL=data:application/json;base64,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