"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataMeshWorkflowRole = void 0;
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const lake_formation_1 = require("../lake-formation");
/**
 * Construct extending IAM Role with managed and inline policies for LF admin. Adds this role as LF admin.
 */
class DataMeshWorkflowRole extends constructs_1.Construct {
    /**
     * Constructs a new instance of the LfAdminRole class.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {RoleProps} props the RoleProps [properties]{@link RoleProps}
     * @access public
     */
    constructor(scope, id) {
        super(scope, id);
        this.role = new aws_iam_1.Role(this, 'Role', {
            assumedBy: new aws_iam_1.CompositePrincipal(new aws_iam_1.ServicePrincipal('states.amazonaws.com'), new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'))
        });
        new aws_iam_1.ManagedPolicy(this, 'WorkflowRolePolicy', {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'lakeformation:*',
                        'glue:GetDatabase',
                        'glue:GetDatabases',
                        'glue:GetTable',
                        'glue:GetTables',
                        'glue:CreateTable',
                        'glue:UpdateTable',
                        'iam:GetRole'
                    ],
                    resources: ['*'],
                    effect: aws_iam_1.Effect.ALLOW,
                }),
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ram:CreateResourceShare'
                    ],
                    resources: ['*'],
                    effect: aws_iam_1.Effect.ALLOW,
                    conditions: {
                        StringLikeIfExists: {
                            'ram:RequestedResourceType': [
                                'glue:Table',
                                'glue:Database',
                                'glue:Catalog'
                            ]
                        }
                    }
                }),
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'ram:UpdateResourceShare',
                        'ram:AssociateResourceShare',
                        'ram:GetResourceShares'
                    ],
                    resources: ['*'],
                    effect: aws_iam_1.Effect.ALLOW,
                    conditions: {
                        StringLike: {
                            'ram:ResourceShareName': [
                                'LakeFormation*'
                            ]
                        }
                    }
                }),
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'glue:PutResourcePolicy',
                        'ram:Get*',
                        'ram:List*'
                    ],
                    resources: ['*'],
                    effect: aws_iam_1.Effect.ALLOW,
                }),
            ],
            roles: [this.role],
        });
        // Add this role to LF admins
        new lake_formation_1.LakeFormationAdmin(this, 'LfAdmin', {
            principal: this.role,
        });
    }
}
exports.DataMeshWorkflowRole = DataMeshWorkflowRole;
//# sourceMappingURL=data:application/json;base64,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