import { Construct } from 'constructs';
import { StateMachine } from 'aws-cdk-lib/aws-stepfunctions';
import { DataDomainWorkflow } from './data-domain-workflow';
/**
 * This CDK Construct creates a TBAC workflow for Producer/Consumer account.
 * It is based on an AWS Step Functions state machine. It has the following steps:
 * * grants permissions to data domain LF tag to workflow role
 * * creates tables in database that is shared via LF tag
 * * creates Resource-Link(s) for database
 *
 * This Step Functions state machine is invoked from the Central Gov. account via EventBridge Event Bus.
 * It is initiatated in {@link DataDomain}, but can be used as a standalone construct.
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { DataDomain } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * const workflowRole = new Role(stack, 'myWorkflowRole', {
 *  assumedBy: ...
 * });
 *
 * new DataDomainTbacWorkflow(this, 'DataDomainWorkflow', {
 *  domainName: 'departmentName',
 *  eventBus: eventBus,
 *  workflowRole: workflowRole,
 *  centralAccountId: '1234567891011',
 *  domainName: 'domainName'
 * });
 * ```
 *
 */
export declare class DataDomainTbacWorkflow extends Construct {
    readonly stateMachine: StateMachine;
    /**
     * Construct a new instance of DataDomainTbacWorkflow.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainWorkflowProps} props the DataDomainWorkflowProps properties
     * @access public
     */
    constructor(scope: Construct, id: string, props: DataDomainWorkflow);
}
