"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataDomainNracWorkflow = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
/**
 * This CDK Construct creates a NRAC workflow for Producer/Consumer account.
 * It is based on an AWS Step Functions state machine. It has the following steps:
 * * checks for AWS RAM invitations
 * * accepts RAM invitations if the source is Central Gov. account
 * * creates AWS Glue Data Catalog Database and tables
 * * creates Resource-Link(s) for created tables
 *
 * This Step Functions state machine is invoked from the Central Gov. account via EventBridge Event Bus.
 * It is initiatated in {@link DataDomain}, but can be used as a standalone construct.
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { DataDomain } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * const workflowRole = new Role(stack, 'myWorkflowRole', {
 *  assumedBy: ...
 * });
 *
 * new DataDomainNracWorkflow(this, 'DataDomainWorkflow', {
 *  eventBus: eventBus,
 *  workflowRole: workflowRole,
 *  centralAccountId: '1234567891011',
 *  domainName: 'domainName'
 * });
 * ```
 *
 */
class DataDomainNracWorkflow extends constructs_1.Construct {
    /**
     * Construct a new instance of DataDomainNracWorkflow.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainWorkflow} props the DataDomainWorkflowProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Task to check for existing RAM invitations
        const getRamInvitations = new aws_stepfunctions_tasks_1.CallAwsService(this, 'GetResourceShareInvitations', {
            service: 'ram',
            action: 'getResourceShareInvitations',
            iamResources: ['*'],
            parameters: {},
            resultPath: '$.taskresult',
        });
        // Task to accept RAM share invitation
        const acceptRamShare = new aws_stepfunctions_tasks_1.CallAwsService(this, 'AcceptResourceShareInvitation', {
            service: 'ram',
            action: 'acceptResourceShareInvitation',
            iamResources: ['*'],
            parameters: {
                'ResourceShareInvitationArn.$': '$.ram_share.ResourceShareInvitationArn',
            },
            resultPath: '$.Response',
            resultSelector: {
                'Status.$': '$.ResourceShareInvitation.Status',
            },
        });
        const createLocalDatabase = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createLocalDatabase', {
            service: 'glue',
            action: 'createDatabase',
            iamResources: ['*'],
            parameters: {
                'DatabaseInput': {
                    'Name.$': '$.detail.database_name'
                },
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const grantCreateTable = new aws_stepfunctions_tasks_1.CallAwsService(this, 'grantCreateTable', {
            service: 'lakeformation',
            action: 'grantPermissions',
            iamResources: ['*'],
            parameters: {
                'Permissions': [
                    'ALL'
                ],
                'Principal': {
                    'DataLakePrincipalIdentifier': props.workflowRole.roleArn
                },
                'Resource': {
                    'Database': {
                        'Name.$': '$.detail.database_name'
                    },
                }
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        });
        // Task to create a resource-link for shared table from central gov accunt
        const createResourceLink = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createResourceLink', {
            service: 'glue',
            action: 'createTable',
            iamResources: ['*'],
            parameters: {
                'DatabaseName.$': '$.database_name',
                'TableInput': {
                    'Name.$': "States.Format('rl-{}', $.table_name)",
                    'TargetTable': {
                        'CatalogId': props.centralAccountId,
                        'DatabaseName.$': '$.central_database_name',
                        'Name.$': '$.table_name',
                    },
                },
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        // Trigger crawler workflow
        const triggerCrawler = new aws_stepfunctions_tasks_1.EventBridgePutEvents(this, 'triggerCrawler', {
            entries: [{
                    detail: aws_stepfunctions_1.TaskInput.fromObject({
                        'database_name': aws_stepfunctions_1.JsonPath.stringAt("$.database_name"),
                        'table_names': aws_stepfunctions_1.JsonPath.stringAt("$.table_names"),
                        'central_account_id': aws_stepfunctions_1.JsonPath.stringAt("$.central_account_id"),
                        'central_database_name': aws_stepfunctions_1.JsonPath.stringAt("$.central_database_name"),
                        'lf_access_mode': aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"),
                    }),
                    detailType: 'triggerCrawler',
                    eventBus: props.eventBus,
                    source: 'com.central.stepfunction',
                }]
        });
        // Pass task to finish the workflow
        const finishWorkflow = new aws_stepfunctions_1.Pass(this, 'finishWorkflow');
        const rlMapTask = new aws_stepfunctions_1.Map(this, 'forEachTable', {
            itemsPath: '$.table_names',
            parameters: {
                'central_database_name.$': '$.central_database_name',
                'database_name.$': '$.database_name',
                'table_name.$': '$$.Map.Item.Value'
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        rlMapTask.iterator(createResourceLink);
        rlMapTask.next(new aws_stepfunctions_1.Choice(this, 'thisAccountIsProducer')
            .when(aws_stepfunctions_1.Condition.stringEquals('$.producer_acc_id', aws_cdk_lib_1.Aws.ACCOUNT_ID), triggerCrawler)
            .otherwise(finishWorkflow));
        // Task to iterate over RAM shares and check if there are PENDING invites from the central account
        const ramMapTask = new aws_stepfunctions_1.Map(this, 'forEachRamInvitation', {
            itemsPath: '$.taskresult.ResourceShareInvitations',
            parameters: {
                'ram_share.$': '$$.Map.Item.Value',
                'central_account_id.$': '$.account',
                'central_database_name.$': '$.detail.central_database_name',
                'database_name.$': '$.detail.database_name',
                'table_names.$': '$.detail.table_names',
                'producer_acc_id.$': '$.detail.producer_acc_id',
                'lf_access_mode.$': '$.detail.lf_access_mode',
            },
            resultPath: '$.map_result',
            outputPath: '$.map_result.[?(@.central_account_id)]',
        });
        const invitationAccepted = new aws_stepfunctions_1.Pass(this, 'invitationAccepted', {
            result: aws_stepfunctions_1.Result.fromObject({ 'Status': 'ACCEPTED' }),
            resultPath: '$.Response'
        });
        ramMapTask.iterator(new aws_stepfunctions_1.Choice(this, 'isInvitationPending')
            .when(aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.stringEqualsJsonPath('$.ram_share.SenderAccountId', '$.central_account_id'), aws_stepfunctions_1.Condition.stringEquals('$.ram_share.Status', 'PENDING')), acceptRamShare)
            .when(aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.stringEqualsJsonPath('$.ram_share.SenderAccountId', '$.central_account_id'), aws_stepfunctions_1.Condition.stringEquals('$.ram_share.Status', 'ACCEPTED')), invitationAccepted)
            .otherwise(new aws_stepfunctions_1.Pass(this, 'notValidPass', {
            result: aws_stepfunctions_1.Result.fromObject({})
        })));
        ramMapTask.next(new aws_stepfunctions_1.Choice(this, 'shareAccepted', { outputPath: '$[0]' })
            .when(aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.isPresent('$[0]'), aws_stepfunctions_1.Condition.stringEquals('$[0].Response.Status', 'ACCEPTED')), rlMapTask).otherwise(finishWorkflow));
        // Avoid possible delays in between RAM share time and EventBridge event time 
        const initWait = new aws_stepfunctions_1.Wait(this, 'InitWait', {
            time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(5))
        });
        createLocalDatabase.addCatch(grantCreateTable, {
            errors: ['Glue.AlreadyExistsException'],
            resultPath: '$.Exception',
        }).next(grantCreateTable).next(ramMapTask);
        // Create Log group for this state machine
        const logGroup = new aws_logs_1.LogGroup(this, 'Workflow', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            logGroupName: '/aws/vendedlogs/data-mesh/nrac-workflow',
        });
        logGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // State Machine workflow to accept RAM share and create resource-link for a shared table
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'nracStateMachine', {
            definition: initWait.next(getRamInvitations).next(new aws_stepfunctions_1.Choice(this, 'resourceShareInvitationsEmpty')
                .when(aws_stepfunctions_1.Condition.isPresent('$.taskresult.ResourceShareInvitations[0]'), createLocalDatabase)
                .otherwise(finishWorkflow)),
            role: props.workflowRole,
            logs: {
                destination: logGroup,
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
    }
}
exports.DataDomainNracWorkflow = DataDomainNracWorkflow;
//# sourceMappingURL=data:application/json;base64,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