import { Construct } from 'constructs';
import { SfnStateMachine } from 'aws-cdk-lib/aws-events-targets';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { IBucket } from 'aws-cdk-lib/aws-s3';
import { IEventBus } from 'aws-cdk-lib/aws-events';
/**
 * Properties for the DataDomainCrawler Construct
 */
export interface DataDomainCrawlerProps {
    /**
     * LF Admin Role
     */
    readonly workflowRole: IRole;
    /**
     * Data Products S3 bucket
     */
    readonly dataProductsBucket: IBucket;
    /**
     * Data Products S3 bucket prefix
     */
    readonly dataProductsPrefix: string;
    /**
    * Data domain name
    */
    readonly domainName: string;
    /**
    * Event Bus in Data Domain
    */
    readonly eventBus: IEventBus;
}
/**
 * This CDK Construct creates a AWS Glue Crawler workflow in Data Domain account.
 * It uses AWS Step Functions state machine to orchestrate the workflow:
 * * creates and destroys the crawler upon execution
 *
 * It is triggered via Amazon EventBridge Rule upon successful execution of DataDomainWorkflow state machine {@link DataDomainWorkflow}.
 * This construct is initiatied in {@link DataDomain} but can be used as a standalone construct.
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { DataDomainCrawler } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = cdk.App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * # See {@link DataDomain}
 *
 * new DataDomainCrawler(this, 'DataDomainCrawler', {
 *  workflowRole: workflowRole,
 *  dataProductsBucket: dataProductsBucket,
 *  domainName: 'domainName'
 * });
 * ```
 *
 */
export declare class DataDomainCrawler extends Construct {
    readonly stateMachine: SfnStateMachine;
    readonly crawlerRole: IRole;
    /**
     * Construct a new instance of DataDomainCrawler.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainCrawlerProps} props the DataDomainCrawlerProps properties
     * @access public
     */
    constructor(scope: Construct, id: string, props: DataDomainCrawlerProps);
}
