"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchReplayer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
/**
 * Replay the data in the given PartitionedDataset.
 *
 * It will dump files into the `sinkBucket` based on the given `frequency`.
 * The computation is in a Step Function with two Lambda steps.
 *
 * 1. resources/lambdas/find-file-paths
 * Read the manifest file and output a list of S3 file paths within that batch time range
 *
 * 2. resources/lambdas/write-in-batch
 * Take a file path, filter only records within given time range, adjust the the time with offset to
 * make it looks like just being generated. Then write the output to the `sinkBucket`
 *
 * Usage example:
 * ```typescript
 *
 * const myBucket = new Bucket(stack, "MyBucket")
 *
 * new BatchReplayer(stack, "WebSalesReplayer", {
 *   dataset: PreparedDataset.RETAIL_1_GB_WEB_SALE,
 *   s3BucketSink: myBucket
 *   s3ObjectKeySink: 'some-prefix',
 *   frequency: 120,
 *   outputFileMaxSizeInBytes: 10000000,
 * });
 * ```
 *
 * :warnning: **If the Bucket is encrypted with KMS, the Key must be managed by this stack.
 */
class BatchReplayer extends constructs_1.Construct {
    /**
     * Constructs a new instance of the BatchReplayer construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {BatchReplayerProps} props the BatchReplayer [properties]{@link BatchReplayerProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Used to force S3 bucket auto cleaning after deletion of this
        this.node.addDependency(props.sinkBucket);
        this.dataset = props.dataset;
        this.frequency = props.frequency?.toSeconds() || 60;
        this.sinkBucket = props.sinkBucket;
        this.sinkObjectKey = this.sinkObjectKey ? `${this.sinkObjectKey}/${this.dataset.tableName}` : this.dataset.tableName;
        this.outputFileMaxSizeInBytes = props.outputFileMaxSizeInBytes || 100 * 1024 * 1024; //Default to 100 MB
        const dataWranglerLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'PandasLayer', `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:336392948345:layer:AWSDataWrangler-Python39:1`);
        const manifestBucketName = this.dataset.manifestLocation.bucketName;
        const manifestObjectKey = this.dataset.manifestLocation.objectKey;
        const dataBucketName = this.dataset.location.bucketName;
        const dataObjectKey = this.dataset.location.objectKey;
        const findFilePathsFnPolicy = [
            new aws_iam_1.PolicyStatement({
                actions: [
                    's3:GetObject',
                    's3:ListBucket',
                ],
                resources: [
                    `arn:aws:s3:::${dataBucketName}/${dataObjectKey}/*`,
                    `arn:aws:s3:::${dataBucketName}/${dataObjectKey}-manifest.csv`,
                    `arn:aws:s3:::${dataBucketName}`,
                ],
            }),
        ];
        /**
         * Find all paths within the time range from the manifest file
         */
        const findFilePathsFn = new pre_bundled_function_1.PreBundledFunction(this, 'FindFilePath', {
            memorySize: 1024,
            codePath: 'data-generator/resources/lambdas/find-file-paths',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'find-file-paths.handler',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            layers: [dataWranglerLayer],
            lambdaPolicyStatements: findFilePathsFnPolicy,
        });
        const findFilePathsFnTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'FindFilePathFnTask', {
            lambdaFunction: findFilePathsFn,
            payload: aws_stepfunctions_1.TaskInput.fromObject({
                frequency: this.frequency,
                manifestFileBucket: manifestBucketName,
                manifestFileKey: manifestObjectKey,
                triggerTime: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Input.time'),
                offset: this.dataset.offset,
            }),
            // Retry on 500 error on invocation with an interval of 2 sec with back-off rate 2, for 6 times
            retryOnServiceExceptions: true,
            outputPath: '$.Payload',
        });
        const writeInBatchFnPolicy = [
            new aws_iam_1.PolicyStatement({
                actions: [
                    's3:GetObject',
                    's3:ListBucket',
                ],
                resources: [
                    `arn:aws:s3:::${dataBucketName}/${dataObjectKey}/*`,
                    `arn:aws:s3:::${dataBucketName}`,
                ],
            }),
        ];
        /**
         * Rewrite data
         */
        const writeInBatchFn = new pre_bundled_function_1.PreBundledFunction(this, 'WriteInBatch', {
            memorySize: 3008,
            codePath: 'data-generator/resources/lambdas/write-in-batch',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'write-in-batch.handler',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            layers: [dataWranglerLayer],
            lambdaPolicyStatements: writeInBatchFnPolicy,
        });
        // grant permissions to write to the bucket and to use the KMS key
        this.sinkBucket.grantWrite(writeInBatchFn, `${this.sinkObjectKey}/*`);
        const writeInBatchFnTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'WriteInBatchFnTask', {
            lambdaFunction: writeInBatchFn,
            payload: aws_stepfunctions_1.TaskInput.fromObject({
                // Array from the last step to be mapped
                outputFileIndex: aws_stepfunctions_1.JsonPath.stringAt('$.index'),
                filePath: aws_stepfunctions_1.JsonPath.stringAt('$.filePath'),
                // For calculating the start/end time
                frequency: this.frequency,
                triggerTime: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Input.time'),
                offset: this.dataset.offset,
                // For file processing
                dateTimeColumnToFilter: this.dataset.dateTimeColumnToFilter,
                dateTimeColumnsToAdjust: this.dataset.dateTimeColumnsToAdjust,
                sinkPath: this.sinkBucket.s3UrlForObject(this.sinkObjectKey),
                outputFileMaxSizeInBytes: 20480,
            }),
            // Retry on 500 error on invocation with an interval of 2 sec with back-off rate 2, for 6 times
            retryOnServiceExceptions: true,
            outputPath: '$.Payload',
        });
        // Use "Map" step to write each filePath parallelly
        const writeInBatchMapTask = new aws_stepfunctions_1.Map(this, 'WriteInBatchMapTask', {
            itemsPath: aws_stepfunctions_1.JsonPath.stringAt('$.filePaths'),
            parameters: {
                index: aws_stepfunctions_1.JsonPath.stringAt('$$.Map.Item.Index'),
                filePath: aws_stepfunctions_1.JsonPath.stringAt('$$.Map.Item.Value'),
            },
        });
        writeInBatchMapTask.iterator(writeInBatchFnTask);
        // Overarching Step Function StateMachine
        const batchReplayStepFn = new aws_stepfunctions_1.StateMachine(this, 'BatchReplayStepFn', {
            definition: findFilePathsFnTask.next(writeInBatchMapTask),
            timeout: aws_cdk_lib_1.Duration.minutes(20),
            logs: {
                destination: new aws_logs_1.LogGroup(this, 'LogGroup', {
                    retention: aws_logs_1.RetentionDays.ONE_WEEK,
                    logGroupName: `/aws/batch-replayer/${this.dataset.tableName}`,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
        new aws_events_1.Rule(this, 'BatchReplayStepFnTrigger', {
            schedule: aws_events_1.Schedule.cron({ minute: `0/${Math.ceil(this.frequency / 60)}` }),
            targets: [new aws_events_targets_1.SfnStateMachine(batchReplayStepFn, {})],
        });
    }
}
exports.BatchReplayer = BatchReplayer;
_a = JSII_RTTI_SYMBOL_1;
BatchReplayer[_a] = { fqn: "aws-analytics-reference-architecture.BatchReplayer", version: "2.8.1" };
//# sourceMappingURL=data:application/json;base64,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