"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventsRuleToStepFunction = void 0;
const events = require("@aws-cdk/aws-events");
const defaults = require("@aws-solutions-constructs/core");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
class EventsRuleToStepFunction extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the EventsRuleToStepFunction class.
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        [this.stateMachine, this.stateMachineLogGroup] = defaults.buildStateMachine(this, props.stateMachineProps);
        // Create an IAM role for Events to start the State Machine
        const eventsRole = new iam.Role(this, 'EventsRuleRole', {
            assumedBy: new iam.ServicePrincipal('events.amazonaws.com')
        });
        // Grant the start execution permission to the Events service
        this.stateMachine.grantStartExecution(eventsRole);
        // Setup the Events target
        const stateMachine = {
            bind: () => ({
                id: '',
                arn: this.stateMachine.stateMachineArn,
                role: eventsRole
            })
        };
        // Defaults props for the Events
        const defaultEventsRuleProps = defaults.DefaultEventsRuleProps([stateMachine]);
        // Override the defaults with the user provided props
        const eventsRuleProps = core_2.overrideProps(defaultEventsRuleProps, props.eventRuleProps, true);
        // Create the Events Rule for the State Machine
        this.eventsRule = new events.Rule(this, 'EventsRule', eventsRuleProps);
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for State Machine
            this.cloudwatchAlarms = defaults.buildStepFunctionCWAlarms(this, this.stateMachine);
        }
    }
}
exports.EventsRuleToStepFunction = EventsRuleToStepFunction;
//# sourceMappingURL=data:application/json;base64,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