"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Yaml = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs = require("fs");
const os = require("os");
const path = require("path");
const YAML = require("yaml");
const MAX_DOWNLOAD_BUFFER = 10 * 1024 * 1024;
// Set default YAML schema to 1.1. This ensures saved YAML is backward compatible with other parsers, such as PyYAML
// It also ensures that octal numbers in the form `0775` will be parsed
// correctly on YAML load. (see https://github.com/eemeli/yaml/issues/205)
YAML.defaultOptions.version = '1.1';
/**
 * YAML utilities.
 *
 * @stability stable
 */
class Yaml {
    /**
     * Utility class.
     */
    constructor() {
        return;
    }
    /**
     * @deprecated use `stringify(doc[, doc, ...])`
     */
    static formatObjects(docs) {
        return this.stringify(...docs);
    }
    /**
     * Saves a set of objects as a multi-document YAML file.
     *
     * @param filePath The output path.
     * @param docs The set of objects.
     * @stability stable
     */
    static save(filePath, docs) {
        const data = this.stringify(...docs);
        fs.writeFileSync(filePath, data, { encoding: 'utf8' });
    }
    /**
     * Stringify a document (or multiple documents) into YAML.
     *
     * We convert undefined values to null, but ignore any documents that are
     * undefined.
     *
     * @param docs A set of objects to convert to YAML.
     * @returns a YAML string. Multiple docs are separated by `---`.
     * @stability stable
     */
    static stringify(...docs) {
        return docs.map(r => r === undefined ? '\n' : YAML.stringify(r, { keepUndefined: true, lineWidth: 0 })).join('---\n');
    }
    /**
     * Saves a set of YAML documents into a temp file (in /tmp).
     *
     * @param docs the set of documents to save.
     * @returns the path to the temporary file
     * @stability stable
     */
    static tmp(docs) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk8s-'));
        const filePath = path.join(tmpdir, 'temp.yaml');
        Yaml.save(filePath, docs);
        return filePath;
    }
    /**
     * Downloads a set of YAML documents (k8s manifest for example) from a URL or a file and returns them as javascript objects.
     *
     * Empty documents are filtered out.
     *
     * @param urlOrFile a URL of a file path to load from.
     * @returns an array of objects, each represents a document inside the YAML
     * @stability stable
     */
    static load(urlOrFile) {
        const body = loadurl(urlOrFile);
        const objects = YAML.parseAllDocuments(body);
        const result = new Array();
        for (const obj of objects.map(x => x.toJSON())) {
            // skip empty documents
            if (obj === undefined) {
                continue;
            }
            if (obj === null) {
                continue;
            }
            if (Array.isArray(obj) && obj.length === 0) {
                continue;
            }
            if (typeof (obj) === 'object' && Object.keys(obj).length === 0) {
                continue;
            }
            result.push(obj);
        }
        return result;
    }
}
exports.Yaml = Yaml;
_a = JSII_RTTI_SYMBOL_1;
Yaml[_a] = { fqn: "cdk8s.Yaml", version: "2.1.2" };
/**
 * Loads a url (or file) and returns the contents.
 * This method spawns a child process in order to perform an http call synchronously.
 */
function loadurl(url) {
    const script = path.join(__dirname, '_loadurl.js');
    return child_process_1.execFileSync(process.execPath, [script, url], {
        encoding: 'utf-8',
        maxBuffer: MAX_DOWNLOAD_BUFFER,
    }).toString();
}
//# sourceMappingURL=data:application/json;base64,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