/**
 * Represents the amount of digital storage.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 *
 * @stability stable
 */
export declare class Size {
    /**
     * Create a Storage representing an amount kibibytes.
     *
     * 1 KiB = 1024 bytes
     *
     * @stability stable
     */
    static kibibytes(amount: number): Size;
    /**
     * Create a Storage representing an amount mebibytes.
     *
     * 1 MiB = 1024 KiB
     *
     * @stability stable
     */
    static mebibytes(amount: number): Size;
    /**
     * Create a Storage representing an amount gibibytes.
     *
     * 1 GiB = 1024 MiB
     *
     * @stability stable
     */
    static gibibytes(amount: number): Size;
    /**
     * Create a Storage representing an amount tebibytes.
     *
     * 1 TiB = 1024 GiB
     *
     * @stability stable
     */
    static tebibytes(amount: number): Size;
    /**
     * Create a Storage representing an amount pebibytes.
     *
     * 1 PiB = 1024 TiB
     *
     * @stability stable
     */
    static pebibyte(amount: number): Size;
    private readonly amount;
    private readonly unit;
    private constructor();
    /**
     * Return this storage as a total number of kibibytes.
     *
     * @stability stable
     */
    toKibibytes(opts?: SizeConversionOptions): number;
    /**
     * Return this storage as a total number of mebibytes.
     *
     * @stability stable
     */
    toMebibytes(opts?: SizeConversionOptions): number;
    /**
     * Return this storage as a total number of gibibytes.
     *
     * @stability stable
     */
    toGibibytes(opts?: SizeConversionOptions): number;
    /**
     * Return this storage as a total number of tebibytes.
     *
     * @stability stable
     */
    toTebibytes(opts?: SizeConversionOptions): number;
    /**
     * Return this storage as a total number of pebibytes.
     *
     * @stability stable
     */
    toPebibytes(opts?: SizeConversionOptions): number;
}
/**
 * Rounding behaviour when converting between units of `Size`.
 *
 * @stability stable
 */
export declare enum SizeRoundingBehavior {
    /**
     * Fail the conversion if the result is not an integer.
     *
     * @stability stable
     */
    FAIL = 0,
    /**
     * If the result is not an integer, round it to the closest integer less than the result.
     *
     * @stability stable
     */
    FLOOR = 1,
    /**
     * Don't round.
     *
     * Return even if the result is a fraction.
     *
     * @stability stable
     */
    NONE = 2
}
/**
 * Options for how to convert time to a different unit.
 *
 * @stability stable
 */
export interface SizeConversionOptions {
    /**
     * How conversions should behave when it encounters a non-integer result.
     *
     * @default SizeRoundingBehavior.FAIL
     * @stability stable
     */
    readonly rounding?: SizeRoundingBehavior;
}
