"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Names = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const constructs_1 = require("constructs");
const MAX_LEN = 63;
const VALIDATE = /^[0-9a-z-]+$/;
const VALIDATE_LABEL_VALUE = /^(([0-9a-zA-Z][0-9a-zA-Z-_.]*)?[0-9a-zA-Z])?$/;
const HASH_LEN = 8;
/**
 * Utilities for generating unique and stable names.
 *
 * @stability stable
 */
class Names {
    /* istanbul ignore next */
    constructor() {
        return;
    }
    /**
     * Generates a unique and stable name compatible DNS_LABEL from RFC-1123 from a path.
     *
     * The generated name will:
     *   - contain at most 63 characters
     *   - contain only lowercase alphanumeric characters or ‘-’
     *   - start with an alphanumeric character
     *   - end with an alphanumeric character
     *
     * The generated name will have the form:
     *   <comp0>-<comp1>-..-<compN>-<short-hash>
     *
     * Where <comp> are the path components (assuming they are is separated by
     * "/").
     *
     * Note that if the total length is longer than 63 characters, we will trim
     * the first components since the last components usually encode more meaning.
     *
     * @param scope The construct for which to render the DNS label.
     * @param options Name options.
     * @stability stable
     * @link https://tools.ietf.org/html/rfc1123
     * @throws if any of the components do not adhere to naming constraints or
     * length.
     */
    static toDnsLabel(scope, options = {}) {
        var _b, _c, _d, _e;
        const maxLen = (_b = options.maxLen) !== null && _b !== void 0 ? _b : MAX_LEN;
        const delim = (_c = options.delimiter) !== null && _c !== void 0 ? _c : '-';
        const include_hash = (_d = options.includeHash) !== null && _d !== void 0 ? _d : true;
        if (maxLen < HASH_LEN && include_hash) {
            throw new Error(`minimum max length for object names is ${HASH_LEN} (required for hash)`);
        }
        const node = constructs_1.Node.of(scope);
        let components = node.path.split('/');
        components.push(...(_e = options.extra) !== null && _e !== void 0 ? _e : []);
        // special case: if we only have one component in our path and it adheres to DNS_NAME, we don't decorate it
        if (components.length === 1 && VALIDATE.test(components[0]) && components[0].length <= maxLen) {
            return components[0];
        }
        // okay, now we need to normalize all components to adhere to DNS_NAME and append the hash of the full path.
        components = components.map(c => normalizeToDnsName(c, maxLen));
        if (include_hash) {
            components.push(calcHash(node, HASH_LEN));
        }
        return toHumanForm(components, delim, maxLen);
    }
    /**
     * Generates a unique and stable name compatible label key name segment and label value from a path.
     *
     * The name segment is required and must be 63 characters or less, beginning
     * and ending with an alphanumeric character ([a-z0-9A-Z]) with dashes (-),
     * underscores (_), dots (.), and alphanumerics between.
     *
     * Valid label values must be 63 characters or less and must be empty or
     * begin and end with an alphanumeric character ([a-z0-9A-Z]) with dashes
     * (-), underscores (_), dots (.), and alphanumerics between.
     *
     * The generated name will have the form:
     *   <comp0><delim><comp1><delim>..<delim><compN><delim><short-hash>
     *
     * Where <comp> are the path components (assuming they are is separated by
     * "/").
     *
     * Note that if the total length is longer than 63 characters, we will trim
     * the first components since the last components usually encode more meaning.
     *
     * @param scope The construct for which to render the DNS label.
     * @param options Name options.
     * @stability stable
     * @link https://kubernetes.io/docs/concepts/overview/working-with-objects/labels/#syntax-and-character-set
     * @throws if any of the components do not adhere to naming constraints or
     * length.
     */
    static toLabelValue(scope, options = {}) {
        var _b, _c, _d, _e;
        const maxLen = (_b = options.maxLen) !== null && _b !== void 0 ? _b : MAX_LEN;
        const delim = (_c = options.delimiter) !== null && _c !== void 0 ? _c : '-';
        const include_hash = (_d = options.includeHash) !== null && _d !== void 0 ? _d : true;
        if (maxLen < HASH_LEN && include_hash) {
            throw new Error(`minimum max length for label is ${HASH_LEN} (required for hash)`);
        }
        if (/[^0-9a-zA-Z-_.]/.test(delim)) {
            throw new Error('delim should not contain "[^0-9a-zA-Z-_.]"');
        }
        const node = constructs_1.Node.of(scope);
        let components = node.path.split('/');
        components.push(...(_e = options.extra) !== null && _e !== void 0 ? _e : []);
        // special case: if we only have one component in our path and it adheres to DNS_NAME, we don't decorate it
        if (components.length === 1 && VALIDATE_LABEL_VALUE.test(components[0]) && components[0].length <= maxLen) {
            return components[0];
        }
        // okay, now we need to normalize all components to adhere to label and append the hash of the full path.
        components = components.map(c => normalizeToLabelValue(c, maxLen));
        if (include_hash) {
            components.push(calcHash(node, HASH_LEN));
        }
        const result = toHumanForm(components, delim, maxLen);
        // slicing might let '-', '_', '.' be in the start of the result.
        return result.replace(/^[^0-9a-zA-Z]+/, '');
    }
}
exports.Names = Names;
_a = JSII_RTTI_SYMBOL_1;
Names[_a] = { fqn: "cdk8s.Names", version: "2.1.2" };
function omitDuplicates(value, index, components) {
    return value !== components[index - 1];
}
function omitDefaultChild(value, _, __) {
    return value.toLowerCase() !== 'resource' && value.toLowerCase() !== 'default';
}
function toHumanForm(components, delim, maxLen) {
    return components.reverse()
        .filter(omitDuplicates)
        .join('/')
        .slice(0, maxLen)
        .split('/')
        .reverse()
        .filter(x => x)
        .join(delim)
        .split(delim)
        .filter(x => x)
        .filter(omitDefaultChild)
        .join(delim);
}
function normalizeToDnsName(c, maxLen) {
    return c
        .toLocaleLowerCase() // lower case
        .replace(/[^0-9a-zA-Z-_.]/g, '') // remove non-allowed characters
        .substr(0, maxLen); // trim to maxLength
}
function calcHash(node, maxLen) {
    if (process.env.CDK8S_LEGACY_HASH) {
        const hash = crypto.createHash('sha256');
        hash.update(node.path);
        return hash.digest('hex').slice(0, maxLen);
    }
    return node.addr.substring(0, HASH_LEN);
}
function normalizeToLabelValue(c, maxLen) {
    return c
        .replace(/[^0-9a-zA-Z-_.]/g, '') // remove non-allowed characters
        .substr(0, maxLen); // trim to maxLength
}
//# sourceMappingURL=data:application/json;base64,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