"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonPatch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fjp = require("fast-json-patch");
/**
 * Utility for applying RFC-6902 JSON-Patch to a document.
 *
 * Use the the `JsonPatch.apply(doc, ...ops)` function to apply a set of
 * operations to a JSON document and return the result.
 *
 * Operations can be created using the factory methods `JsonPatch.add()`,
 * `JsonPatch.remove()`, etc.
 *
 * @stability stable
 * @example
 *
 * const output = JsonPatch.apply(input,
 *  JsonPatch.replace('/world/hi/there', 'goodbye'),
 *  JsonPatch.add('/world/foo/', 'boom'),
 *  JsonPatch.remove('/hello'));
 */
class JsonPatch {
    constructor(operation) {
        this.operation = operation;
    }
    /**
     * Applies a set of JSON-Patch (RFC-6902) operations to `document` and returns the result.
     *
     * @param document The document to patch.
     * @param ops The operations to apply.
     * @returns The result document
     * @stability stable
     */
    static apply(document, ...ops) {
        const result = fjp.applyPatch(document, ops.map(o => o._toJson()));
        return result.newDocument;
    }
    /**
     * Adds a value to an object or inserts it into an array.
     *
     * In the case of an
     * array, the value is inserted before the given index. The - character can be
     * used instead of an index to insert at the end of an array.
     *
     * @stability stable
     * @example
     *
     * JsonPatch.add('/biscuits/1', { "name": "Ginger Nut" })
     */
    static add(path, value) { return new JsonPatch({ op: 'add', path, value }); }
    /**
     * Removes a value from an object or array.
     *
     * @stability stable
     * @example
     *
     * JsonPatch.remove('/biscuits/0')
     */
    static remove(path) { return new JsonPatch({ op: 'remove', path }); }
    /**
     * Replaces a value.
     *
     * Equivalent to a “remove” followed by an “add”.
     *
     * @stability stable
     * @example
     *
     * JsonPatch.replace('/biscuits/0/name', 'Chocolate Digestive')
     */
    static replace(path, value) { return new JsonPatch({ op: 'replace', path, value }); }
    /**
     * Copies a value from one location to another within the JSON document.
     *
     * Both
     * from and path are JSON Pointers.
     *
     * @stability stable
     * @example
     *
     * JsonPatch.copy('/biscuits/0', '/best_biscuit')
     */
    static copy(from, path) { return new JsonPatch({ op: 'copy', from, path }); }
    /**
     * Moves a value from one location to the other.
     *
     * Both from and path are JSON Pointers.
     *
     * @stability stable
     * @example
     *
     * JsonPatch.move('/biscuits', '/cookies')
     */
    static move(from, path) { return new JsonPatch({ op: 'move', from, path }); }
    /**
     * Tests that the specified value is set in the document.
     *
     * If the test fails,
     * then the patch as a whole should not apply.
     *
     * @stability stable
     * @example
     *
     * JsonPatch.test('/best_biscuit/name', 'Choco Leibniz')
     */
    static test(path, value) { return new JsonPatch({ op: 'test', path, value }); }
    /**
     * Returns the JSON representation of this JSON patch operation.
     *
     * @internal
     */
    _toJson() {
        return this.operation;
    }
}
exports.JsonPatch = JsonPatch;
_a = JSII_RTTI_SYMBOL_1;
JsonPatch[_a] = { fqn: "cdk8s.JsonPatch", version: "2.1.2" };
//# sourceMappingURL=data:application/json;base64,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