"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyVertex = exports.DependencyGraph = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
/**
 * Represents the dependency graph for a given Node.
 *
 * This graph includes the dependency relationships between all nodes in the
 * node (construct) sub-tree who's root is this Node.
 *
 * Note that this means that lonely nodes (no dependencies and no dependants) are also included in this graph as
 * childless children of the root node of the graph.
 *
 * The graph does not include cross-scope dependencies. That is, if a child on the current scope depends on a node
 * from a different scope, that relationship is not represented in this graph.
 *
 * @stability stable
 */
class DependencyGraph {
    /**
     * @stability stable
     */
    constructor(node) {
        this._fosterParent = new DependencyVertex();
        const nodes = {};
        function putVertex(construct) {
            nodes[constructs_1.Node.of(construct).path] = new DependencyVertex(construct);
        }
        function getVertex(construct) {
            return nodes[constructs_1.Node.of(construct).path];
        }
        // create all vertices of the graph.
        for (const n of node.findAll()) {
            putVertex(n);
        }
        const deps = [];
        for (const child of node.findAll()) {
            for (const dep of child.node.dependencies) {
                deps.push({ source: child, target: dep });
            }
        }
        // create all the edges of the graph.
        for (const dep of deps) {
            if (!getVertex(dep.target)) {
                // dont cross scope boundaries.
                // since charts only renders its own children, this is ok and
                // has the benefit of simplifying the graph. we should reconsider this behavior when moving
                // to a more general purpose use-case.
                continue;
            }
            const sourceDepNode = getVertex(dep.source);
            const targetDepNode = getVertex(dep.target);
            sourceDepNode.addChild(targetDepNode);
        }
        // create the root.
        for (const n of Object.values(nodes)) {
            if (n.inbound.length === 0) {
                // orphans are dependency roots. lets adopt them!
                this._fosterParent.addChild(n);
            }
        }
    }
    /**
     * Returns the root of the graph.
     *
     * Note that this vertex will always have `null` as its `.value` since it is an artifical root
     * that binds all the connected spaces of the graph.
     *
     * @stability stable
     */
    get root() {
        return this._fosterParent;
    }
    /**
     * @see Vertex.topology()
     * @stability stable
     */
    topology() {
        return this._fosterParent.topology();
    }
}
exports.DependencyGraph = DependencyGraph;
_a = JSII_RTTI_SYMBOL_1;
DependencyGraph[_a] = { fqn: "cdk8s.DependencyGraph", version: "2.1.2" };
/**
 * Represents a vertex in the graph.
 *
 * The value of each vertex is an `IConstruct` that is accessible via the `.value` getter.
 *
 * @stability stable
 */
class DependencyVertex {
    /**
     * @stability stable
     */
    constructor(value = undefined) {
        this._children = new Set();
        this._parents = new Set();
        this._value = value;
    }
    /**
     * Returns the IConstruct this graph vertex represents.
     *
     * `null` in case this is the root of the graph.
     *
     * @stability stable
     */
    get value() {
        return this._value;
    }
    /**
     * Returns the children of the vertex (i.e dependencies).
     *
     * @stability stable
     */
    get outbound() {
        return Array.from(this._children);
    }
    /**
     * Returns the parents of the vertex (i.e dependants).
     *
     * @stability stable
     */
    get inbound() {
        return Array.from(this._parents);
    }
    /**
     * Returns a topologically sorted array of the constructs in the sub-graph.
     *
     * @stability stable
     */
    topology() {
        const found = new Set();
        const topology = [];
        function visit(n) {
            for (const c of n.outbound) {
                visit(c);
            }
            if (!found.has(n)) {
                topology.push(n);
                found.add(n);
            }
        }
        visit(this);
        return topology.filter(d => d.value).map(d => d.value);
    }
    /**
     * Adds a vertex as a dependency of the current node.
     *
     * Also updates the parents of `dep`, so that it contains this node as a parent.
     *
     * This operation will fail in case it creates a cycle in the graph.
     *
     * @param dep The dependency.
     * @stability stable
     */
    addChild(dep) {
        const cycle = dep.findRoute(this);
        if (cycle.length !== 0) {
            cycle.push(dep);
            throw new Error(`Dependency cycle detected: ${cycle.filter(d => d.value).map(d => constructs_1.Node.of(d.value).path).join(' => ')}`);
        }
        this._children.add(dep);
        dep.addParent(this);
    }
    addParent(dep) {
        this._parents.add(dep);
    }
    findRoute(dst) {
        const route = [];
        visit(this);
        return route;
        function visit(n) {
            route.push(n);
            let found = false;
            for (const c of n.outbound) {
                if (c === dst) {
                    route.push(c);
                    return true;
                }
                found = visit(c);
            }
            if (!found) {
                route.pop();
            }
            return found;
        }
    }
}
exports.DependencyVertex = DependencyVertex;
_b = JSII_RTTI_SYMBOL_1;
DependencyVertex[_b] = { fqn: "cdk8s.DependencyVertex", version: "2.1.2" };
//# sourceMappingURL=data:application/json;base64,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