"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = exports.YamlOutputType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const constructs_1 = require("constructs");
const api_object_1 = require("./api-object");
const chart_1 = require("./chart");
const dependency_1 = require("./dependency");
const names_1 = require("./names");
const yaml_1 = require("./yaml");
/**
 * The method to divide YAML output into files.
 *
 * @stability stable
 */
var YamlOutputType;
(function (YamlOutputType) {
    YamlOutputType[YamlOutputType["FILE_PER_APP"] = 0] = "FILE_PER_APP";
    YamlOutputType[YamlOutputType["FILE_PER_CHART"] = 1] = "FILE_PER_CHART";
    YamlOutputType[YamlOutputType["FILE_PER_RESOURCE"] = 2] = "FILE_PER_RESOURCE";
    YamlOutputType[YamlOutputType["FOLDER_PER_CHART_FILE_PER_RESOURCE"] = 3] = "FOLDER_PER_CHART_FILE_PER_RESOURCE";
})(YamlOutputType = exports.YamlOutputType || (exports.YamlOutputType = {}));
/**
 * Represents a cdk8s application.
 *
 * @stability stable
 */
class App extends constructs_1.Construct {
    /**
     * Defines an app.
     *
     * @param props configuration options.
     * @stability stable
     */
    constructor(props = {}) {
        var _b, _c, _d;
        super(undefined, '');
        this.outdir = (_c = (_b = props.outdir) !== null && _b !== void 0 ? _b : process.env.CDK8S_OUTDIR) !== null && _c !== void 0 ? _c : 'dist';
        this.yamlOutputType = (_d = props.yamlOutputType) !== null && _d !== void 0 ? _d : YamlOutputType.FILE_PER_CHART;
    }
    /**
     * Synthesize a single chart.
     *
     * Each element returned in the resulting array represents a different ApiObject
     * in the scope of the chart.
     *
     * Note that the returned array order is important. It is determined by the various dependencies between
     * the constructs in the chart, where the first element is the one without dependencies, and so on...
     *
     * @returns An array of JSON objects.
     * @param chart the chart to synthesize.
     * @internal
     */
    static _synthChart(chart) {
        const app = App.of(chart);
        // we must prepare the entire app before synthesizing the chart
        // because the dependency inference happens on the app level.
        resolveDependencies(app);
        // validate the app since we want to call onValidate of the relevant constructs.
        // note this will also call onValidate on constructs from possibly different charts,
        // but thats ok too since we no longer treat constructs as a self-contained synthesis unit.
        validate(app);
        return chartToKube(chart).map(obj => obj.toJson());
    }
    static of(c) {
        const scope = constructs_1.Node.of(c).scope;
        if (!scope) {
            // the app is the only construct without a scope.
            return c;
        }
        return App.of(scope);
    }
    /**
     * Returns all the charts in this app, sorted topologically.
     *
     * @stability stable
     */
    get charts() {
        const isChart = (x) => x instanceof chart_1.Chart;
        return new dependency_1.DependencyGraph(constructs_1.Node.of(this))
            .topology()
            .filter(isChart);
    }
    /**
     * Synthesizes all manifests to the output directory.
     *
     * @stability stable
     */
    synth() {
        fs.mkdirSync(this.outdir, { recursive: true });
        // Since we plan on removing the distributed synth mechanism, we no longer call `Node.synthesize`, but rather simply implement
        // the necessary operations. We do however want to preserve the distributed validation.
        validate(this);
        // this is kind of sucky, eventually I would like the DependencyGraph
        // to be able to answer this question.
        const hasDependantCharts = resolveDependencies(this);
        const charts = this.charts;
        switch (this.yamlOutputType) {
            case YamlOutputType.FILE_PER_APP:
                let apiObjectList = [];
                for (const chart of charts) {
                    apiObjectList.push(...chartToKube(chart));
                }
                if (charts.length > 0) {
                    yaml_1.Yaml.save(path.join(this.outdir, 'app.k8s.yaml'), // There is no "app name", so we just hardcode the file name
                    apiObjectList.map((apiObject) => apiObject.toJson()));
                }
                break;
            case YamlOutputType.FILE_PER_CHART:
                const namer = hasDependantCharts ? new IndexedChartNamer() : new SimpleChartNamer();
                for (const chart of charts) {
                    const chartName = namer.name(chart);
                    const objects = chartToKube(chart);
                    yaml_1.Yaml.save(path.join(this.outdir, chartName), objects.map(obj => obj.toJson()));
                }
                break;
            case YamlOutputType.FILE_PER_RESOURCE:
                for (const chart of charts) {
                    const apiObjects = chartToKube(chart);
                    apiObjects.forEach((apiObject) => {
                        if (!(apiObject === undefined)) {
                            const fileName = `${`${apiObject.kind}.${apiObject.metadata.name}`
                                .replace(/[^0-9a-zA-Z-_.]/g, '')}.k8s.yaml`;
                            yaml_1.Yaml.save(path.join(this.outdir, fileName), [apiObject.toJson()]);
                        }
                    });
                }
                break;
            case YamlOutputType.FOLDER_PER_CHART_FILE_PER_RESOURCE:
                const folderNamer = hasDependantCharts ? new IndexedChartFolderNamer() : new SimpleChartFolderNamer();
                for (const chart of charts) {
                    const chartName = folderNamer.name(chart);
                    const apiObjects = chartToKube(chart);
                    const fullOutDir = path.join(this.outdir, chartName);
                    fs.mkdirSync(fullOutDir, { recursive: true });
                    apiObjects.forEach((apiObject) => {
                        if (!(apiObject === undefined)) {
                            const fileName = `${`${apiObject.kind}.${apiObject.metadata.name}`
                                .replace(/[^0-9a-zA-Z-_.]/g, '')}.k8s.yaml`;
                            yaml_1.Yaml.save(path.join(fullOutDir, fileName), [apiObject.toJson()]);
                        }
                    });
                }
                break;
            default:
                break;
        }
    }
    /**
     * Synthesizes the app into a YAML string.
     *
     * @returns A string with all YAML objects across all charts in this app.
     * @stability stable
     */
    synthYaml() {
        validate(this);
        const charts = this.charts;
        const docs = [];
        for (const chart of charts) {
            const apiObjects = chartToKube(chart);
            docs.push(...apiObjects.map(apiObject => apiObject.toJson()));
        }
        return yaml_1.Yaml.stringify(...docs);
    }
}
exports.App = App;
_a = JSII_RTTI_SYMBOL_1;
App[_a] = { fqn: "cdk8s.App", version: "2.1.2" };
function validate(app) {
    const errors = [];
    for (const child of app.node.findAll()) {
        const childErrors = child.node.validate();
        for (const error of childErrors) {
            errors.push(`[${child.node.path}] ${error}`);
        }
    }
    if (errors.length > 0) {
        throw new Error(`Validation failed with the following errors:\n  ${errors.join('\n  ')}`);
    }
}
function resolveDependencies(app) {
    let hasDependantCharts = false;
    const deps = [];
    for (const child of app.node.findAll()) {
        for (const dep of child.node.dependencies) {
            deps.push({ source: child, target: dep });
        }
    }
    for (const dep of deps) {
        // create explicit api object dependencies from implicit construct dependencies
        const targetApiObjects = constructs_1.Node.of(dep.target).findAll().filter(c => c instanceof api_object_1.ApiObject);
        const sourceApiObjects = constructs_1.Node.of(dep.source).findAll().filter(c => c instanceof api_object_1.ApiObject);
        for (const target of targetApiObjects) {
            for (const source of sourceApiObjects) {
                if (target !== source) {
                    constructs_1.Node.of(source).addDependency(target);
                }
            }
        }
        // create an explicit chart dependency from implicit construct dependencies
        const sourceChart = chart_1.Chart.of(dep.source);
        const targetChart = chart_1.Chart.of(dep.target);
        if (sourceChart !== targetChart) {
            constructs_1.Node.of(sourceChart).addDependency(targetChart);
            hasDependantCharts = true;
        }
    }
    const charts = new dependency_1.DependencyGraph(constructs_1.Node.of(app)).topology()
        .filter(x => x instanceof chart_1.Chart);
    for (const parentChart of charts) {
        for (const childChart of constructs_1.Node.of(parentChart).children.filter(x => x instanceof chart_1.Chart)) {
            // create an explicit chart dependency from nested chart relationships
            constructs_1.Node.of(parentChart).addDependency(childChart);
            hasDependantCharts = true;
        }
    }
    return hasDependantCharts;
}
function chartToKube(chart) {
    return new dependency_1.DependencyGraph(constructs_1.Node.of(chart)).topology()
        .filter(x => x instanceof api_object_1.ApiObject)
        .filter(x => chart_1.Chart.of(x) === chart) // include an object only in its closest parent chart
        .map(x => x);
}
class SimpleChartNamer {
    constructor() {
    }
    name(chart) {
        return `${names_1.Names.toDnsLabel(chart)}.k8s.yaml`;
    }
}
class IndexedChartNamer extends SimpleChartNamer {
    constructor() {
        super();
        this.index = 0;
    }
    name(chart) {
        const name = `${this.index.toString().padStart(4, '0')}-${super.name(chart)}`;
        this.index++;
        return name;
    }
}
class SimpleChartFolderNamer {
    constructor() {
    }
    name(chart) {
        return names_1.Names.toDnsLabel(chart);
    }
}
class IndexedChartFolderNamer extends SimpleChartFolderNamer {
    constructor() {
        super();
        this.index = 0;
    }
    name(chart) {
        const name = `${this.index.toString().padStart(4, '0')}-${super.name(chart)}`;
        this.index++;
        return name;
    }
}
//# sourceMappingURL=data:application/json;base64,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