# Copyright (c) 2022 MIEK
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

'''
Implement helper functions for Serverless Cloud Function products.
'''

import os
import json
import base64
import datetime

def inferred_return_result(
    return_result: str
) -> object:
    '''
    Infer the Python native data type for a given return value.

    Args:
        return_result: Cloud Function return value.
    
    Returns:
        Returns a Python native data type container containing
            the return value.
    
    Raises:
        ValueError: Parameter values are not as expected.
    '''

    if return_result:
        if not isinstance(return_result, str):
            raise ValueError('<return_result> value invalid')
    else:
        return None

    if (return_result[ : 1] == '\"' and
        return_result[-1 : ] == '\"'
    ):
        return return_result[1 : -1]
        
    if return_result == 'true':
        return True
    elif return_result == 'false':
        return False

    try:
        return json.loads(return_result)
    except json.JSONDecodeError:
        return return_result

def local_file_to_base64(
    local_file_path: str
) -> str:
    '''
    Opens the given local file and reads all data using Base64 encoding.

    Args:
        local_file_path: Local file path.
    
    Returns:
        Returns a Base64-encoded string of file data.
    
    Raises:
        ValueError: Parameter values are not as expected.
        FileNotFoundError: The given file path does not exist.
        IOError: An error occurred while reading the given file.
    '''

    if not local_file_path or not isinstance(local_file_path, str):
        raise ValueError('<local_file_path> value invalid')

    with open(local_file_path, mode = 'rb') as local_file_context:
        return base64.b64encode(local_file_context.read()).decode('utf-8')

async def generator_proxy_async(
    async_generator: object
) -> object:
    '''
    Asynchronous generator proxy.

    Args:
        async_generator: Asynchronous generator instance.
    
    Returns:
        Returns the value generated by the asynchronous generator.
            If there is no new value, None is returned.
    
    Raises:
        TypeError: Parameter values are not as expected.
    '''

    async for value in async_generator:
        return value
    
    return None

def format_time_by_timestamp(
    unix_timestamp: int
) -> str:
    '''
    Formats a UNIX timestamp as a time string.

    Note that the given UNIX timestamp must be UTC time.
        This function returns the corresponding time string
        in the Asia/Shanghai time zone.

    Args:
        unix_timestamp: Given a UNIX timestamp value.
    
    Returns:
        Returns a formatted time string.
    
    Raises:
        ValueError: Parameter values are not as expected.
    '''

    if not unix_timestamp or not isinstance(unix_timestamp, int):
        raise ValueError('<unix_timestamp> value invalid')

    return (datetime.datetime.utcfromtimestamp(unix_timestamp) +
        datetime.timedelta(hours = 8)).strftime('%Y-%m-%d %H:%M:%S')

def is_cloud_function_container() -> bool:
    '''
    Whether the current running environment is a
        Serverless Cloud Function container instance.

    Returns:
        Returns true if yes, otherwise false.
    '''

    return os.environ.get('TENCENTCLOUD_RUNENV', None) == 'SCF'

def get_cloud_function_region_id() -> str:
    '''
    Gets the unique identifier of the Region where the current
        Serverless Cloud Function container instance is located.
    
    Please note that this function is only supported when running within
        a Serverless Cloud Function container instance or simulation environment.

    Returns:
        Returns a Region unique identifier string.
    
    Raises:
        EnvironmentError: The current operating environment is not as expected.
    '''

    try:
        return os.environ['TENCENTCLOUD_REGION']
    except KeyError:
        raise EnvironmentError('missing environment variable <TENCENTCLOUD_REGION>')

def get_cloud_function_namespace_name() -> str:
    '''
    Get the name of the namespace where the current serverless cloud
        function is located. If reading the namespace name fails, the
        default namespace name is'default'.
    
    Please note that this function is only supported when running
        within a Serverless Cloud Function container instance or
        simulation environment.

    Returns:
        Returns a string of the namespace name.
    
    Raises:
        EnvironmentError: The current operating environment is not as
            expected.
    '''

    try:
        return os.environ['SCF_NAMESPACE']
    except KeyError:
        return 'default'
