# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['bas_web_map_inventory',
 'bas_web_map_inventory.components',
 'bas_web_map_inventory.resources',
 'bas_web_map_inventory.resources.json_schemas',
 'bas_web_map_inventory.resources.xml_schemas',
 'tests',
 'tests.bas_web_map_inventory',
 'tests.bas_web_map_inventory.conftest']

package_data = \
{'': ['*'], 'tests': ['data/*', 'resources/validate_ogc_capabilities/*']}

install_requires = \
['Flask>=1.1.1,<2.0.0',
 'OWSlib>=0.19.0,<0.20.0',
 'airtable-python-wrapper>=0.12.0,<0.13.0',
 'geoserver-restconfig==1.0.2',
 'inquirer>=2.6.3,<3.0.0',
 'jsonschema>=3.2.0,<4.0.0',
 'lxml==4.5.0',
 'pyproj==2.6.0',
 'python-dotenv>=0.10.3,<0.11.0',
 'sentry-sdk[flask]>=0.13.5,<0.14.0',
 'str2bool>=1.1,<2.0',
 'ulid-py>=0.0.12,<0.0.13']

setup_kwargs = {
    'name': 'bas-web-map-inventory',
    'version': '0.4.0',
    'description': 'Inventory of geospatial layers and web maps provided by the BAS Mapping and Geographic Information Centre (MAGIC), visualised in Airtable.',
    'long_description': '# MAGIC Web Map Inventory\n\nInventory of geospatial layers and web maps provided by the BAS Mapping and Geographic Information Centre (MAGIC),\nvisualised in Airtable.\n\n* [view the inventory in Airtable (Internal)](https://airtable.com/tblr6gwKOLQelMXDv/viwp8xVwcRRJnoIo0?blocks=hide)\n\nSee the [Data model](#data-model) section for more information about what this inventory holds.\n\n**Note:** This project is designed for internal use within MAGIC, but is offered to anyone with similar needs.\n\n## Usage\n\nThis project runs in a container. See the [Setup](#setup) section for setup instructions.\n\nIf running on the BAS central workstations:\n\n```shell\n$ ssh geoweb@bslws01.nerc-bas.ac.uk\n$ web-map-inventory [task]\n```\n\nConfiguration, logs and data output are stored in `/users/geoweb/.config/web-map-inventory/`.\n\nIf any errors occur they will be reported to Sentry and relevant individials alerted by email.\n\n### Commands\n\n#### `version`\n\nReports the current application version.\n\n#### `data fetch`\n\nFetches information about servers, namespaces, repositories, styles, layers and layer groups from servers defined in a\ndata sources file. Fetched information is saved to an output data file.\n\nOptions:\n\n* `-s`, `--data-sources-file-path`:\n    * path to a data sources file\n    * default: `data/sources.json`\n* `-d`, `--data-output-file-path`:\n    * path to a data sources file\n    * default: `data/data.json`\n\n**Note:** Currently this task results in new IDs being generated for each resource, even if it already exists. This will\nlead to resources being removed and re-added unnecessarily but will always remain internally consistent.\n\n#### `data validate`\n\nValidates protocols offered by servers defined in a data sources file (by default `data/sources.json`).\n\nOptions:\n\n* `-s`, `--data-sources-file-path`:\n    * path to a data sources file\n    * default: `data/sources.json`\n* `-i`, `--data-source-identifier`:\n    * identifier of a server in the data sources file\n    * use special value `all` to select all data sources\n* `-p`, `--validation-protocol`:\n    * protocol to validate\n    * default: `wms`\n\n**Note:** Currently this task is limited to the WMS (OGC Web Map Service) protocol.\n\n#### `airtable status`\n\nChecks local items against Airtable to check whether they are up-to-date (current), outdated, missing or orphaned.\n\n#### `airtable sync`\n\nCreates, updates or removes items in Airtable to match local items.\n\n#### `airtable reset`\n\nRemoves all data from Airtable.\n\n### Managing data sources\n\nEach data source is represented as an object in the `server` list in `data/sources.json` [1]. The structure of each\nsource depends on its type. For more general information, see the [Data sources](#data-sources) section.\n\n[1] This file is either in the runtime path created during [Setup](#setup) or `~/.config/web-map-inventory/` on the BAS\ncentral servers).\n\n#### Adding new data sources\n\n**Note:** See [Supported data sources](#supported-data-sources) for currently supported data sources.\n\nOnce added use the [`data fetch` task](#data-fetch).\n\n#### Adding a *GeoServer* data source\n\n| Property   | Required | Data Type | Allowed Values                                                      | Example Value                | Description                          | Notes                         |\n| ---------- | -------- | --------- | ------------------------------------------------------------------- | ---------------------------- | ------------------------------------ | ----------------------------- |\n| `id`       | Yes      | String    | A *ULID* (Universally Unique Lexicographically Sortable Identifier) | `01DRS53XAJNH0TNBW5161B6EWJ` | Unique identifier for server/source  | See below for how to generate |\n| `label`    | Yes      | String    | Any combination of *a-Z*, *A-Z*, *0-9*, *-*, *_*                    | `a-1_A`                      | Using a short, well-known identifier | -                             |\n| `hostname` | Yes      | String    | Any valid hostname                                                  | `example.com`                | -                                    | -                             |\n| `type`     | Yes      | String    | `geoserver`                                                         | *See allowed value*          | -                                    | -                             |\n| `port`     | Yes      | String    | Any valid port number                                               | `8080`                       | -                                    | Usually `80` or `8080`        |\n| `api-path` | Yes      | String    | `/geoserver/rest`                                                   | *See allowed value*          | Defined by GeoServer                 | -                             |\n| `wms-path` | Yes      | String    | `/geoserver/ows?service=wms&version=1.3.0&request=GetCapabilities`  | *See allowed value*          | Defined by GeoServer                 | -                             |\n| `wfs-path` | Yes      | String    | `/geoserver/ows?service=wfs&version=2.0.0&request=GetCapabilities`  | *See allowed value*          | Defined by GeoServer                 | -                             |\n| `username` | Yes      | String    | Any valid GeoServer username                                        | `admin`                      | Usually the GeoServer admin user     | -                             |\n| `password` | Yes      | String    | Password for GeoServer user                                         | `password`                   | Usually the GeoServer admin user     | -                             |\n\n**Note:** Use [ulidgenerator.com](http://ulidgenerator.com) to generate ULIDs manually.\n\nExample:\n\n```json\n{\n  "id": "xxx",\n  "label": "example",\n  "hostname": "example.com",\n  "type": "geoserver",\n  "port": "80",\n  "api-path": "/geoserver/rest",\n  "wms-path": "/geoserver/ows?service=wms&version=1.3.0&request=GetCapabilities",\n  "wfs-path": "/geoserver/ows?service=wfs&version=2.0.0&request=GetCapabilities",\n  "username": "admin",\n  "password": "password"\n}\n```\n\n**Note:** Make sure to include a comma between each source (but don\'t include a trailing comma) - e.g.:\n\n```json\n{\n    "servers": [\n        {\n            "id": "xxx",\n            "label": "existing-server"\n            ...\n        },\n        {\n            "id": "xxx",\n            "label": "new-server"\n            ...\n        }\n    ]\n}\n```\n\n## Implementation\n\nFlask application using the [airtable-python-wrapper](https://airtable-python-wrapper.readthedocs.io) library to\ninteract with the Airtable API.\n\n### Airtable\n\nData is synced to the\n[MAGIC Maps and Layers Inventory](https://airtable.com/tblCoGkVssEe6cs0B/viwjb9FAq2FLx5BL9?blocks=hide) Base in the\n[BAS MAGIC](https://airtable.com/wspXVL8SsiS5hPhob/workspace/billing) Workspace.\n\n### Data model\n\nThis project, an inventory, consists of information held in geospatial services. The data model is intended to be\ngeneric to support different data sources and technologies.\n\nThis data model consists of:\n\n* **Servers**: Represent a source of geospatial information, such as an instance of a technology or a whole platform\n* **Namespaces**: Represent a logical grouping of resources within a server/endpoint\n* **Repositories**: Represent a data source that backs one or more layers\n* **Styles**: Represent a definition for how data in a layer should be represented/presented\n* **Layers**: Represent a logical unit of geospatial information\n* **Layer Groups**: Represent a logical grouping of one or more layers that should be treated as a single, indivisible\n  unit\n\n![Data model visualisation](https://raw.githubusercontent.com/antarctica/web-map-inventory/master/assets/img/data-model.png)\n\nA [JSON Schema](bas_web_map_inventory/resources/json_schemas/data-schema.json) describes this schema. It is used \ninternally for validating data prior to use but is also published for use by others if needed as:\n[data-schema-v1.json](https://metadata-standards.data.bas.ac.uk/bas-web-map-inventory-configuration-schemas/data-resources-schema-v1.json).\n\n### Data sources\n\nData sources are *servers* in the project [Data model](#data-model) and define connection details for APIs and services\neach server type provides for fetching information about components they contain (e.g. listing *layers*).\n\nA data sources file, `data/sources.json`, is used for recording these details. An example is available in\n`data/sources.example.json`. See the [Adding a data source](#adding-new-data-sources) section for more information.\n\nA JSON Schema, `bas_web_map_inventory/resources/json_schemas/data-sources-schema.json`, validates this file.\n\n#### Supported data sources\n\n* GeoServer\n    * Using a combination of its admin API and WMS/WFS OGC endpoints\n\n### Configuration\n\nConfiguration options are set within `bas_web_map_inventory/config.py`.\n\nAll [Options](#configuration-options) are defined in a `Config` base class, with per-environment sub-classes overriding\nand extending these options as needed. The active configuration is set using the `FLASK_ENV` environment variable.\n\nMost options can be [Set using environment variables or files](#setting-configuration-options).\n\n#### Configuration options\n\n| Option                    | Required | Environments | Data Type (Cast) | Source      |  Allowed Values                                                                                                     | Default Value                                                              | Example Value                                                | Description                                                                                                     | Notes                      |\n| ------------------------- | -------- | ------------ | ---------------- | ----------- | ------------------------------------------------------------------------------------------------------------------- | -------------------------------------------------------------------------- | ------------------------------------------------------------ | --------------------------------------------------------------------------------------------------------------- | -------------------------- |\n| `FLASK_APP`               | Yes      | All          | String           | `.flaskenv` | Valid [`FLASK_APP`](https://flask.palletsprojects.com/en/1.1.x/cli/#application-discovery) value                    | `manage.py`                                                                | *See default value*                                          | See [Flask documentation](https://flask.palletsprojects.com/en/1.1.x/cli/#application-discovery)                | -                          |\n| `APP_ENABLE_SENTRY`       | No       | All          | Boolean          | `.flaskenv` | `True`/`False`                                                                                                      | `False` (for *development*/*testing*), `True` (for *staging*/*production*) | `True`                                                       | Feature flag for [Error reporting](#error-reporting)                                                            | -                          |\n| `APP_ENABLE_FILE_LOGGING` | No       | All          | Boolean          | `.flaskenv` | `True`/`False`                                                                                                      | `False`                                                                    | `False`                                                      | Feature flag for writing [Application Logs](#logging) to a file in addition to standard out                     | -                          |\n| `SENTEY_DSN`              | Yes      | Yes          | String           | `.flaskenv` | [Sentry DSN](https://docs.sentry.io/error-reporting/quickstart/?platform=python#configure-the-sdk) for this project | `https://c69a62ee2262460f9bc79c4048ba764f@sentry.io/1832548`               | *See default value*                                          | Sentry [Data Source Name](https://docs.sentry.io/error-reporting/quickstart/?platform=python#configure-the-sdk) | This value is not a secret |\n| `APP_LOG_FILE_PATH`       | No       | All          | String           | `.flaskenv` | Valid file path                                                                                                     | `/var/log/app/app.log`                                                     | `/var/log/app/app.log`                                       | Path to application log file, if enabled                                                                        | -                          |\n| `AIRTABLE_API_KEY`        | Yes      | All          | String           | `.env`      | Valid [AirTable API key](https://support.airtable.com/hc/en-us/articles/219046777-How-do-I-get-my-API-key-)         | -                                                                          | `keyxxxxxxxxxxxxxx`                                          | AirTable API Key                                                                                                | -                          |\n| `AIRTABLE_BASE_ID`        | Yes      | All          | String           | `.env`      | Valid [AirTable Base ID](https://airtable.com/api)                                                                  | -                                                                          | `appxxxxxxxxxxxxxx`                                          | ID of the AirTable Base to populate/use                                                                         | -                          |\n\nOptions are set as strings and then cast to the data type listed above. See\n[Environment variables](#setting-configuration-options) for information about an options \'Source\'.\n\nFlask also has a number of\n[builtin configuration options](https://flask.palletsprojects.com/en/1.1.x/config/#builtin-configuration-values).\n\n#### Setting configuration options\n\nVariable configuration options can be set using environment variables or environment files:\n\n| Source                   | Priority | Purpose                     | Notes                                   |\n| ------------------------ | -------- | --------------------------- | --------------------------------------- |\n| OS environment variables | 1st      | General/Runtime             | -                                       |\n| `.env`                   | 2nd      | Secret/private variables    | Generate by copying `.env.example`      |\n| `.flaskenv`              | 3rd      | Non-secret/public variables | Generate by copying `.flaskenv.example` |\n\n**Note:** these sources are a\n[Flask convention](https://flask.palletsprojects.com/en/1.1.x/cli/#environment-variables-from-dotenv).\n\n### Error tracking\n\nErrors in this service are tracked with Sentry:\n\n* [Sentry dashboard](https://sentry.io/antarctica/web-map-inventory/)\n* [GitLab dashboard](https://gitlab.data.bas.ac.uk/MAGIC/web-map-inventory/error_tracking)\n\nError tracking will be enabled or disabled depending on the environment. It can be manually controlled by setting the\n`APP_ENABLE_SENTRY` variable in `.flaskenv`.\n\n### Logging\n\nLogs for this service are written to *stdout* and a log file, `/var/log/app/app.py`, depending on the environment.\n\nFile based logging can be manually controlled by setting the `APP_ENABLE_FILE_LOGGING` and `APP_LOG_FILE_PATH` \nvariables in `.flaskenv`.\n\n**Note:** If `APP_LOG_FILE_PATH` is changed, the user the container runs as must be granted suitable write permissions.\n\n### XML Catalogue\n\nAn [XML Catalog](https://en.wikipedia.org/wiki/XML_catalog) is used to cache XML files locally (typically XSD\'s for\nschemas). This drastically speeds up XML parsing and removes a dependency on remote endpoints.\n\nXML files in the catalogue are typically stored in `bas_web_map_inventory/resources/xml_schemas/`.\n\nDifferent catalogue files are used for different container variants due to differences in the applications location:\n\n* `:latest`: `./support/xml-schemas/catalogue.xml`\n* `/deploy`: `provisioning/docker/catalog.xml`\n\nIn either case, the catalogue is available within the container at the conventional path, `/etc/xml/catalog`, and will\nbe used automatically by most XML libraries and tools (such as `lxml` and `xmllint`).\n\n## Setup\n\nThe application for this project runs as a Docker container.\n\nOnce setup, see the [Data sources](#managing-data-sources) and [Usage](#usage) sections for how to use and run the\napplication.\n\n**Note:** This project can run locally or on the BAS central workstations using Podman. You will need\naccess to the private BAS Docker Registry (part of [gitlab.data.bas.ac.uk](https://gitlab.data.bas.ac.uk)) and for IT to enable Podman in your user account. Unless noted, `docker` commands listed here can be replaced with `podman`.\n\n```shell\n$ docker login docker-registry.data.bas.ac.uk\n$ docker pull docker-registry.data.bas.ac.uk/magic/web-map-inventory/deploy:stable\n```\n\n**Note:** [Other image tags](https://gitlab.data.bas.ac.uk/MAGIC/web-map-inventory/container_registry) are available if\nyou want to run pre-release versions, or a specific, previous, version.\n\nBefore you can run the container, you will need to create a runtime directory that will live outside of the container.\nYou will need to create the required [Configuration files](#configuration). Any generated output will also be saved\nhere.\n\n```shell\n$ mkdir -p ~/.config/web-map-inventory\n```\n\n### Optional wrapper script\n\nIf using podman, a wrapper script, `support/container-wrapper/podman-wrapper.sh`, is available to make running the\ncontainer easier.\n\nTo use, copy this script and enable it to be executed:\n\n```shell\n$ mkdir ~/bin\n# copy `support/container-wrapper/podman-wrapper.sh` as `~/bin/web-map-inventory`\n$ chmod +x ~/bin/web-map-inventory\n```\n\nThen ensure `~/bin` is part of the user\'s path (use `echo $PATH` to check), if it isn\'t edit the user\'s shell to include\nit (these instructions assume the bash shell and the absolute path to the user\'s home directory is `/home/foo`):\n\n```shell\n$ vi ~/.bash_rc\n# add `export PATH="/home/foo/bin:$PATH" then save the file and reload the user\'s shell\n```\n\nYou should now be able to run `web-map-inventory` to run the container.\n\n### Terraform\n\nTerraform is used to provision resources required to allow JSON Schemas for data resources and data sources to be \naccessed externally.\n\nAccess to the [BAS AWS account](https://gitlab.data.bas.ac.uk/WSF/bas-aws) is needed to provisioning these resources.\n\n**Note:** This provisioning should have already been performed (and applies globally). If changes are made to this \nprovisioning it only needs to be applied once.\n\n```shell\n# start terraform inside a docker container\n$ cd provisioning/terraform\n$ docker-compose run terraform\n# setup terraform\n$ terraform init\n# apply changes\n$ terraform validate\n$ terraform fmt\n$ terraform apply\n# exit container\n$ exit\n$ docker-compose down\n```\n\n#### Terraform remote state\n\nState information for this project is stored remotely using a \n[Backend](https://www.terraform.io/docs/backends/index.html).\n\nSpecifically the [AWS S3](https://www.terraform.io/docs/backends/types/s3.html) backend as part of the \n[BAS Terraform Remote State](https://gitlab.data.bas.ac.uk/WSF/terraform-remote-state) project.\n\nRemote state storage will be automatically initialised when running `terraform init`. Any changes to remote state will\nbe automatically saved to the remote backend, there is no need to push or pull changes.\n\n##### Remote state authentication\n\nPermission to read and/or write remote state information for this project is restricted to authorised users. Contact\nthe [BAS Web & Applications Team](mailto:servicedesk@bas.ac.uk) to request access.\n\nSee the [BAS Terraform Remote State](https://gitlab.data.bas.ac.uk/WSF/terraform-remote-state) project for how these\npermissions to remote state are enforced.\n\n## Development\n\n```shell\n$ git clone https://gitlab.data.bas.ac.uk/MAGIC/web-map-inventory\n$ cd map-layer-index\n```\n\n### Development environment\n\nThe `:latest` container image is used for developing this project. It can run locally using Docker and Docker Compose:\n\n```shell\n$ docker login docker-registry.data.bas.ac.uk\n$ docker-compose pull app\n```\n\nThen create/configure required [Configuration files](#configuration):\n\n```shell\n$ cp .env.example .env\n$ cp .flaskenv.example .flaskenv\n$ cp data/sources.example.json data/sources.json\n```\n\nTo run/test application commands:\n\n```shell\n$ docker-compose run app flask [task]\n```\n\n[1] You will need access to the private BAS Docker Registry (part of\n[gitlab.data.bas.ac.uk](https://gitlab.data.bas.ac.uk)) to pull this image. If you don\'t, you can build the relevant\nimage/tag locally instead.\n\n### Code Style\n\nPEP-8 style and formatting guidelines must be used for this project, with the exception of the 80 character line limit.\n\n[Black](https://github.com/psf/black) is used to ensure compliance, configured in `pyproject.toml`.\n\nBlack can be [integrated](https://black.readthedocs.io/en/stable/editor_integration.html#pycharm-intellij-idea) with a \nrange of editors, such as PyCharm, to perform formatting automatically.\n\nTo apply formatting manually:\n\n```shell\n$ docker-compose run app black bas_web_map_inventory/\n```\n\nTo check compliance manually:\n\n```shell\n$ docker-compose run app black --check bas_web_map_inventory/\n```\n\nChecks are ran automatically in [Continuous Integration](#continuous-integration).\n\n### Python version\n\nWhen upgrading to a new version of Python, ensure the following are also checked and updated where needed:\n\n* `Dockerfile`:\n    * base stage image (e.g. `FROM python:3.X-alpine as base` to `FROM python:3.Y-alpine as base`)\n    * pre-compiled wheels (e.g. `https://.../linux_x86_64/cp3Xm/lxml-4.5.0-cp3X-cp3X-linux_x86_64.whl` to\n     `http://.../linux_x86_64/cp3Ym/lxml-4.5.0-cp3Y-cp3Y-linux_x86_64.whl`)\n* `provisioning/docker/Dockerfile`:\n    * base stage image (e.g. `FROM python:3.X-alpine as base` to `FROM python:3.Y-alpine as base`)\n    * pre-compiled wheels (e.g. `http://.../linux_x86_64/cp3Xm/lxml-4.5.0-cp3X-cp3X-linux_x86_64.whl` to\n     `http://.../linux_x86_64/cp3Ym/lxml-4.5.0-cp3Y-cp3Y-linux_x86_64.whl`)\n* `provisioning/docker/catalog.xml`:\n    * update the path to the Python package (e.g. `file://.../lib/python3.X/site-packages/...` to \n      `file://.../lib/python3.Y/site-packages/...`)\n* `pyproject.toml`\n    * `[tool.poetry.dependencies]`\n        * `python` (e.g. `python = "^3.X"` to `python = "^3.Y"`)\n    * `[tool.black]`\n        * `target-version` (e.g. `target-version = [\'py3X\']` to `target-version = [\'py3Y\']`)\n\n### Dependencies\n\nPython dependencies for this project are managed with [Poetry](https://python-poetry.org) in `pyproject.toml`.\n\nThe development container image installs both runtime and development dependencies. Deployment images only install\nruntime dependencies.\n\nNon-code files, such as static files, can also be included in the [Python package](#python-package) using the\n`include` key in `pyproject.toml`.\n\nTo add a new (development) dependency:\n\n```shell\n$ docker-compose run app ash\n$ poetry add [dependency] (--dev)\n```\n\nThen rebuild the development container and push to GitLab (GitLab will rebuild other images automatically as needed):\n\n```shell\n$ docker-compose build app\n$ docker-compose push app\n```\n\n### Static security scanning\n\nTo ensure the security of this API, source code is checked against [Bandit](https://github.com/PyCQA/bandit) for issues\nsuch as not sanitising user inputs or using weak cryptography. Bandit is configured in `.bandit`.\n\n**Warning:** Bandit is a static analysis tool and can\'t check for issues that are only be detectable when running the\napplication. As with all security tools, Bandit is an aid for spotting common mistakes, not a guarantee of secure code.\n\nTo run checks manually:\n\n```shell\n$ docker-compose run app bandit -r .\n```\n\nChecks are ran automatically in [Continuous Integration](#continuous-integration).\n\n### Logging\n\nUse the Flask default logger. For example:\n\n```python\napp.logger.info(\'Log message\')\n```\n\nWhen outside of a route/command use `current_app`:\n\n```python\nfrom flask import current_app\n\ncurrent_app.logger.info(\'Log message\')\n```\n\n### File paths\n\nUse Python\'s [`pathlib`](https://docs.python.org/3.6/library/pathlib.html) library for managing file paths.\n\nWhere displaying a file path to the user ensure it is always absolute to aid in debugging:\n\n```python\nfrom pathlib import Path\n\nfoo_path = Path("foo.txt")\nprint(f"foo_path is: {str(foo_path.absolute())}")\n```\n\n### JSON Schemas\n\nJSON Schema\'s can be developed using [jsonschemavalidator.net](https://www.jsonschemavalidator.net).\n\n### XML Catalogue additions\n\nIf new functionality is added that depends on XML files, such as XSDs, it is *strongly* recommended to add them to the\n[XML catalogue](#xml-catalogue), especially where they are used in tests.\n\nOnce added, you will need to rebuild and push the project Docker image (see the [Dependencies](#dependencies) section\nfor more information).\n\n### Editor support\n\n#### PyCharm\n\nA run/debug configuration, *App*, is included in the project.\n\n### Testing\n\nAll code in the `bas_web_map_inventory` module must be covered by tests, defined in `tests/`. This project uses\n[PyTest](https://docs.pytest.org/en/latest/) which should be ran in a random order using\n[pytest-random-order](https://pypi.org/project/pytest-random-order/).\n\nTo run tests manually from the command line:\n\n```shell\n$ docker-compose run app -e FLASK_ENV=testing app pytest --random-order\n```\n\nTo run tests manually using PyCharm, use the included *App (Integration)* run/debug configuration.\n\nTests are ran automatically in [Continuous Integration](#continuous-integration).\n\n#### Test coverage\n\n[pytest-cov](https://pypi.org/project/pytest-cov/) is used to measure test coverage.\n\nTo prevent noise, `.coveragerc` is used to omit empty `__init__.py` files from reports.\n\nTo measure coverage manually:\n\n```shell\n$ docker-compose run -e FLASK_ENV=testing app pytest --cov=bas_web_map_inventory --cov-fail-under=100 --cov-report=html .\n```\n\n[Continuous Integration](#continuous-integration) will check coverage automatically and fail if less than 100%.\n\n#### Continuous Integration\n\nAll commits will trigger a Continuous Integration process using GitLab\'s CI/CD platform, configured in `.gitlab-ci.yml`.\n\n## Deployment\n\n### Python package\n\nThis project is distributed as a Python package, hosted in [PyPi](https://pypi.org/project/bas-web-map-inventory).\n\nSource and binary packages are built and published automatically using\n[Poetry](https://python-poetry.org/docs/cli/#publish) in [Continuous Delivery](#continuous-deployment).\n\nPackage versions are determined automatically using the `support/python-packaging/parse_version.py` script.\n\n### Docker image\n\nThe project [Python package](#python-package) is available as a Docker/OCI image, hosted in the private BAS Docker\nRegistry (part of [gitlab.data.bas.ac.uk](https://gitlab.data.bas.ac.uk)).\n\n[Continuous Delivery](#continuous-deployment) will automatically:\n\n* build a `/deploy:latest` image for commits to the *master* branch\n* build a `/deploy:release-stable` and `/deploy:release-[release]` image for tags\n* deploy new images to the BAS central workstations (by running `podman pull [image]` from the workstations)\n\n### Continuous Deployment\n\nAll commits will trigger a Continuous Deployment process using GitLab\'s CI/CD platform, configured in `.gitlab-ci.yml`.\n\n## Release procedure\n\nFor all releases:\n\n1. create a release branch\n2. close release in `CHANGELOG.md`\n3. push changes, merge the release branch into `master` and tag with version\n\n## Feedback\n\nThe maintainer of this project is the BAS Mapping and Geographic Information Centre (MAGIC), they can be contacted at:\n[servicedesk@bas.ac.uk](mailto:servicedesk@bas.ac.uk).\n\n## Issue tracking\n\nThis project uses issue tracking, see the\n[Issue tracker](https://gitlab.data.bas.ac.uk/MAGIC/web-map-inventory/issues) for more information.\n\n**Note:** Read & write access to this issue tracker is restricted. Contact the project maintainer to request access.\n\n## License\n\n© UK Research and Innovation (UKRI), 2019 - 2020, British Antarctic Survey.\n\nYou may use and re-use this software and associated documentation files free of charge in any format or medium, under\nthe terms of the Open Government Licence v3.0.\n\nYou may obtain a copy of the Open Government Licence at http://www.nationalarchives.gov.uk/doc/open-government-licence/\n',
    'author': 'Felix Fennell',
    'author_email': 'felnne@bas.ac.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/antarctica/web-map-inventory',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
