"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxGpuBuildImage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecr = require("@aws-cdk/aws-ecr");
const core = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const project_1 = require("./project");
const mappingName = 'AwsDeepLearningContainersRepositoriesAccounts';
/**
 * A CodeBuild GPU image running Linux.
 *
 * This class has public constants that represent the most popular GPU images from AWS Deep Learning Containers.
 *
 * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
 * @stability stable
 */
class LinuxGpuBuildImage {
    constructor(repositoryName, tag, account) {
        this.repositoryName = repositoryName;
        this.account = account;
        /**
         * The type of build environment.
         *
         * @stability stable
         */
        this.type = 'LINUX_GPU_CONTAINER';
        /**
         * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @stability stable
         */
        this.defaultComputeType = project_1.ComputeType.LARGE;
        /**
         * The type of principal that CodeBuild will use to pull this build Docker image.
         *
         * @stability stable
         */
        this.imagePullPrincipalType = project_1.ImagePullPrincipalType.SERVICE_ROLE;
        this.accountExpression = account !== null && account !== void 0 ? account : core.Fn.findInMap(mappingName, core.Aws.REGION, 'repositoryAccount');
        this.imageId = `${this.accountExpression}.dkr.ecr.${core.Aws.REGION}.${core.Aws.URL_SUFFIX}/${repositoryName}:${tag}`;
    }
    /**
     * Returns a Linux GPU build image from AWS Deep Learning Containers.
     *
     * @param repositoryName the name of the repository, for example "pytorch-inference".
     * @param tag the tag of the image, for example "1.5.0-gpu-py36-cu101-ubuntu16.04".
     * @param account the AWS account ID where the DLC repository for this region is hosted in.
     * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
     * @stability stable
     */
    static awsDeepLearningContainersImage(repositoryName, tag, account) {
        return new LinuxGpuBuildImage(repositoryName, tag, account);
    }
    /**
     * Function that allows the build image access to the construct tree.
     *
     * @stability stable
     */
    bind(scope, project, _options) {
        if (!this.account) {
            const scopeStack = core.Stack.of(scope);
            // Unfortunately, the account IDs of the DLC repositories are not the same in all regions.
            // Because of that, use a (singleton) Mapping to find the correct account
            if (!scopeStack.node.tryFindChild(mappingName)) {
                const mapping = {};
                // get the accounts from the region-info module
                const region2Accounts = region_info_1.RegionInfo.regionMap(region_info_1.FactName.DLC_REPOSITORY_ACCOUNT);
                for (const [region, account] of Object.entries(region2Accounts)) {
                    mapping[region] = { repositoryAccount: account };
                }
                new core.CfnMapping(scopeStack, mappingName, { mapping });
            }
        }
        const repository = ecr.Repository.fromRepositoryAttributes(scope, 'AwsDlcRepositoryCodeBuild', {
            repositoryName: this.repositoryName,
            repositoryArn: ecr.Repository.arnForLocalRepository(this.repositoryName, scope, this.accountExpression),
        });
        repository.grantPull(project);
        return {};
    }
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @stability stable
     */
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== project_1.ComputeType.LARGE) {
            ret.push(`GPU images only support ComputeType '${project_1.ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    /**
     * Make a buildspec to run the indicated script.
     *
     * @stability stable
     */
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxGpuBuildImage = LinuxGpuBuildImage;
_a = JSII_RTTI_SYMBOL_1;
LinuxGpuBuildImage[_a] = { fqn: "@aws-cdk/aws-codebuild.LinuxGpuBuildImage", version: "1.94.0" };
/**
 * Tensorflow 1.14.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_14_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.14.0-gpu-py36-cu100-ubuntu16.04');
/**
 * Tensorflow 1.15.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.0-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 1.15.2 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.2-gpu-py37-cu100-ubuntu18.04');
/**
 * Tensorflow 1.15.2 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '1.15.2-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 2.0.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.0-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 2.0.1 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.1-gpu-py36-cu100-ubuntu18.04');
/**
 * Tensorflow 2.1.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/**
 * Tensorflow 2.1.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/**
 * Tensorflow 2.2.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_2_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.2.0-gpu-py37-cu101-ubuntu18.04');
/**
 * PyTorch 1.2.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_2_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.2.0-gpu-py36-cu100-ubuntu16.04');
/**
 * PyTorch 1.3.1 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_3_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.3.1-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.4.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.4.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.5.0 GPU training image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/**
 * PyTorch 1.5.0 GPU inference image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/**
 * MXNet 1.4.1 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_MXNET_1_4_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.4.1-gpu-py36-cu100-ubuntu16.04');
/**
 * MXNet 1.6.0 GPU image from AWS Deep Learning Containers.
 *
 * @stability stable
 */
LinuxGpuBuildImage.DLC_MXNET_1_6_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.6.0-gpu-py36-cu101-ubuntu16.04');
//# sourceMappingURL=data:application/json;base64,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