"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubSourceAction = exports.GitHubTrigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * If and how the GitHub source action should be triggered.
 */
var GitHubTrigger;
(function (GitHubTrigger) {
    GitHubTrigger["NONE"] = "None";
    GitHubTrigger["POLL"] = "Poll";
    GitHubTrigger["WEBHOOK"] = "WebHook";
})(GitHubTrigger = exports.GitHubTrigger || (exports.GitHubTrigger = {}));
/**
 * Source that is provided by a GitHub repository.
 */
class GitHubSourceAction extends action_1.Action {
    /**
     *
     */
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.SOURCE,
            owner: 'ThirdParty',
            provider: 'GitHub',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /**
     * The variables emitted by this action.
     */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
            commitUrl: this.variableExpression('CommitUrl'),
        };
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     */
    bound(scope, stage, _options) {
        if (!this.props.trigger || this.props.trigger === GitHubTrigger.WEBHOOK) {
            new codepipeline.CfnWebhook(scope, 'WebhookResource', {
                authentication: 'GITHUB_HMAC',
                authenticationConfiguration: {
                    secretToken: this.props.oauthToken.toString(),
                },
                filters: [
                    {
                        jsonPath: '$.ref',
                        matchEquals: 'refs/heads/{Branch}',
                    },
                ],
                targetAction: this.actionProperties.actionName,
                targetPipeline: stage.pipeline.pipelineName,
                targetPipelineVersion: 1,
                registerWithThirdParty: true,
            });
        }
        return {
            configuration: {
                Owner: this.props.owner,
                Repo: this.props.repo,
                Branch: this.props.branch || 'master',
                OAuthToken: this.props.oauthToken.toString(),
                PollForSourceChanges: this.props.trigger === GitHubTrigger.POLL,
            },
        };
    }
}
exports.GitHubSourceAction = GitHubSourceAction;
//# sourceMappingURL=data:application/json;base64,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