"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 *
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 *
 * @stability stable
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    CodeCommitTrigger["NONE"] = "None";
    CodeCommitTrigger["POLL"] = "Poll";
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 *
 * @stability stable
 */
class CodeCommitSourceAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        var _a;
        const branch = (_a = props.branch) !== null && _a !== void 0 ? _a : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /**
     * The variables emitted by this action.
     *
     * @stability stable
     */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline, {
                    eventRole: this.props.eventRole,
                }),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node);
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : `-${this.branch}-`;
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic291cmNlLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNvdXJjZS1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsMERBQTBEO0FBQzFELHVEQUF1RDtBQUN2RCx3Q0FBd0M7QUFDeEMsd0NBQXdEO0FBQ3hELHNDQUFtQztBQUNuQyxzQ0FBaUQ7Ozs7Ozs7O0FBTWpELElBQVksaUJBaUJYO0FBakJELFdBQVksaUJBQWlCO0lBSzNCLGtDQUFhLENBQUE7SUFLYixrQ0FBYSxDQUFBO0lBTWIsc0NBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQWpCVyxpQkFBaUIsR0FBakIseUJBQWlCLEtBQWpCLHlCQUFpQixRQWlCNUI7Ozs7OztBQStERCxNQUFhLHNCQUF1QixTQUFRLGVBQU07Ozs7SUFJaEQsWUFBWSxLQUFrQzs7UUFDNUMsTUFBTSxNQUFNLFNBQUcsS0FBSyxDQUFDLE1BQU0sbUNBQUksUUFBUSxDQUFDO1FBQ3hDLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDWCxNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxDQUFDLENBQUM7U0FDakU7UUFFRCxLQUFLLENBQUM7WUFDSixHQUFHLEtBQUs7WUFDUixRQUFRLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDMUIsUUFBUSxFQUFFLFlBQVksQ0FBQyxjQUFjLENBQUMsTUFBTTtZQUM1QyxRQUFRLEVBQUUsWUFBWTtZQUN0QixjQUFjLEVBQUUsNkJBQW9CLEVBQUU7WUFDdEMsT0FBTyxFQUFFLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQztTQUN4QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztRQUNyQixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztJQUNyQixDQUFDOzs7Ozs7SUFHRCxJQUFXLFNBQVM7UUFDbEIsT0FBTztZQUNMLGNBQWMsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUM7WUFDekQsVUFBVSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxZQUFZLENBQUM7WUFDakQsVUFBVSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxZQUFZLENBQUM7WUFDakQsYUFBYSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlLENBQUM7WUFDdkQsUUFBUSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLENBQUM7WUFDN0MsYUFBYSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlLENBQUM7U0FDeEQsQ0FBQztJQUNKLENBQUM7Ozs7Ozs7O0lBRVMsS0FBSyxDQUFDLE1BQWlCLEVBQUUsS0FBMEIsRUFBRSxPQUF1QztRQUVwRyxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTO1lBQ2xELElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxLQUFLLGlCQUFpQixDQUFDLE1BQU0sQ0FBQztRQUNsRCxJQUFJLFdBQVcsRUFBRTtZQUNmLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDNUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRTtnQkFDdEMsTUFBTSxFQUFFLElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFO29CQUMvQyxTQUFTLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTO2lCQUNoQyxDQUFDO2dCQUNGLFFBQVEsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUM7YUFDeEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCwwRUFBMEU7UUFDMUUsNkRBQTZEO1FBQzdELE9BQU8sQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU1QyxvSEFBb0g7UUFDcEgsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQy9DLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQztZQUNoRCxPQUFPLEVBQUU7Z0JBQ1Asc0JBQXNCO2dCQUN0QixzQkFBc0I7Z0JBQ3RCLDBCQUEwQjtnQkFDMUIsbUNBQW1DO2dCQUNuQyxnQ0FBZ0M7YUFDakM7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLE9BQU87WUFDTCxhQUFhLEVBQUU7Z0JBQ2IsY0FBYyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLGNBQWM7Z0JBQ3BELFVBQVUsRUFBRSxJQUFJLENBQUMsTUFBTTtnQkFDdkIsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssaUJBQWlCLENBQUMsSUFBSTthQUNwRTtTQUNGLENBQUM7SUFDSixDQUFDO0lBRU8sZUFBZSxDQUFDLEtBQTBCO1FBQ2hELE1BQU0sTUFBTSxHQUFHLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ25DLElBQUksU0FBUyxHQUFHLEVBQUUsQ0FBQztZQUNuQixJQUFJLE9BQU8sR0FBRyxDQUFDLENBQUM7WUFDaEIsR0FBRztnQkFDRCxTQUFTLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsTUFBTSxHQUFHLE9BQU8sRUFBRSxDQUFDLENBQUM7Z0JBQzFELE9BQU8sSUFBSSxDQUFDLENBQUM7YUFDZCxRQUFRLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLEtBQUssU0FBUyxFQUFFO1lBQzNFLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO2FBQU07WUFDTCxNQUFNLHFCQUFxQixHQUFHLElBQUksQ0FBQyxNQUFNLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDO1lBQ2pGLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsTUFBTSxHQUFHLHFCQUFxQixFQUFFLENBQUMsQ0FBQztTQUNwRTtJQUNILENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxhQUFxQjtRQUM3QyxPQUFPLEdBQUcsYUFBYSxXQUFXLENBQUM7SUFDckMsQ0FBQztDQUNGO0FBN0ZELHdEQTZGQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVjb21taXQgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVjb21taXQnO1xuaW1wb3J0ICogYXMgY29kZXBpcGVsaW5lIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnO1xuaW1wb3J0ICogYXMgdGFyZ2V0cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzLXRhcmdldHMnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBOYW1lcywgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEFjdGlvbiB9IGZyb20gJy4uL2FjdGlvbic7XG5pbXBvcnQgeyBzb3VyY2VBcnRpZmFjdEJvdW5kcyB9IGZyb20gJy4uL2NvbW1vbic7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gQ29kZUNvbW1pdFRyaWdnZXIge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBOT05FID0gJ05vbmUnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIFBPTEwgPSAnUG9sbCcsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIEVWRU5UUyA9ICdFdmVudHMnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ29kZUNvbW1pdFNvdXJjZVZhcmlhYmxlcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlcG9zaXRvcnlOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBicmFuY2hOYW1lOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhdXRob3JEYXRlOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29tbWl0dGVyRGF0ZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb21taXRJZDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29tbWl0TWVzc2FnZTogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ29kZUNvbW1pdFNvdXJjZUFjdGlvblByb3BzIGV4dGVuZHMgY29kZXBpcGVsaW5lLkNvbW1vbkF3c0FjdGlvblByb3BzIHtcbiAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3V0cHV0OiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3Q7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnJhbmNoPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdHJpZ2dlcj86IENvZGVDb21taXRUcmlnZ2VyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlcG9zaXRvcnk6IGNvZGVjb21taXQuSVJlcG9zaXRvcnk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGV2ZW50Um9sZT86IGlhbS5JUm9sZTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBDb2RlQ29tbWl0U291cmNlQWN0aW9uIGV4dGVuZHMgQWN0aW9uIHtcbiAgcHJpdmF0ZSByZWFkb25seSBicmFuY2g6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBwcm9wczogQ29kZUNvbW1pdFNvdXJjZUFjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBDb2RlQ29tbWl0U291cmNlQWN0aW9uUHJvcHMpIHtcbiAgICBjb25zdCBicmFuY2ggPSBwcm9wcy5icmFuY2ggPz8gJ21hc3Rlcic7XG4gICAgaWYgKCFicmFuY2gpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIidicmFuY2gnIHBhcmFtZXRlciBjYW5ub3QgYmUgYW4gZW1wdHkgc3RyaW5nXCIpO1xuICAgIH1cblxuICAgIHN1cGVyKHtcbiAgICAgIC4uLnByb3BzLFxuICAgICAgcmVzb3VyY2U6IHByb3BzLnJlcG9zaXRvcnksXG4gICAgICBjYXRlZ29yeTogY29kZXBpcGVsaW5lLkFjdGlvbkNhdGVnb3J5LlNPVVJDRSxcbiAgICAgIHByb3ZpZGVyOiAnQ29kZUNvbW1pdCcsXG4gICAgICBhcnRpZmFjdEJvdW5kczogc291cmNlQXJ0aWZhY3RCb3VuZHMoKSxcbiAgICAgIG91dHB1dHM6IFtwcm9wcy5vdXRwdXRdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5icmFuY2ggPSBicmFuY2g7XG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldCB2YXJpYWJsZXMoKTogQ29kZUNvbW1pdFNvdXJjZVZhcmlhYmxlcyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHJlcG9zaXRvcnlOYW1lOiB0aGlzLnZhcmlhYmxlRXhwcmVzc2lvbignUmVwb3NpdG9yeU5hbWUnKSxcbiAgICAgIGJyYW5jaE5hbWU6IHRoaXMudmFyaWFibGVFeHByZXNzaW9uKCdCcmFuY2hOYW1lJyksXG4gICAgICBhdXRob3JEYXRlOiB0aGlzLnZhcmlhYmxlRXhwcmVzc2lvbignQXV0aG9yRGF0ZScpLFxuICAgICAgY29tbWl0dGVyRGF0ZTogdGhpcy52YXJpYWJsZUV4cHJlc3Npb24oJ0NvbW1pdHRlckRhdGUnKSxcbiAgICAgIGNvbW1pdElkOiB0aGlzLnZhcmlhYmxlRXhwcmVzc2lvbignQ29tbWl0SWQnKSxcbiAgICAgIGNvbW1pdE1lc3NhZ2U6IHRoaXMudmFyaWFibGVFeHByZXNzaW9uKCdDb21taXRNZXNzYWdlJyksXG4gICAgfTtcbiAgfVxuXG4gIHByb3RlY3RlZCBib3VuZChfc2NvcGU6IENvbnN0cnVjdCwgc3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2UsIG9wdGlvbnM6IGNvZGVwaXBlbGluZS5BY3Rpb25CaW5kT3B0aW9ucyk6XG4gIGNvZGVwaXBlbGluZS5BY3Rpb25Db25maWcge1xuICAgIGNvbnN0IGNyZWF0ZUV2ZW50ID0gdGhpcy5wcm9wcy50cmlnZ2VyID09PSB1bmRlZmluZWQgfHxcbiAgICAgIHRoaXMucHJvcHMudHJpZ2dlciA9PT0gQ29kZUNvbW1pdFRyaWdnZXIuRVZFTlRTO1xuICAgIGlmIChjcmVhdGVFdmVudCkge1xuICAgICAgY29uc3QgZXZlbnRJZCA9IHRoaXMuZ2VuZXJhdGVFdmVudElkKHN0YWdlKTtcbiAgICAgIHRoaXMucHJvcHMucmVwb3NpdG9yeS5vbkNvbW1pdChldmVudElkLCB7XG4gICAgICAgIHRhcmdldDogbmV3IHRhcmdldHMuQ29kZVBpcGVsaW5lKHN0YWdlLnBpcGVsaW5lLCB7XG4gICAgICAgICAgZXZlbnRSb2xlOiB0aGlzLnByb3BzLmV2ZW50Um9sZSxcbiAgICAgICAgfSksXG4gICAgICAgIGJyYW5jaGVzOiBbdGhpcy5icmFuY2hdLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gdGhlIEFjdGlvbiB3aWxsIHdyaXRlIHRoZSBjb250ZW50cyBvZiB0aGUgR2l0IHJlcG9zaXRvcnkgdG8gdGhlIEJ1Y2tldCxcbiAgICAvLyBzbyBpdHMgUm9sZSBuZWVkcyB3cml0ZSBwZXJtaXNzaW9ucyB0byB0aGUgUGlwZWxpbmUgQnVja2V0XG4gICAgb3B0aW9ucy5idWNrZXQuZ3JhbnRSZWFkV3JpdGUob3B0aW9ucy5yb2xlKTtcblxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlY29tbWl0L2xhdGVzdC91c2VyZ3VpZGUvYXV0aC1hbmQtYWNjZXNzLWNvbnRyb2wtcGVybWlzc2lvbnMtcmVmZXJlbmNlLmh0bWwjYWEtYWNwXG4gICAgb3B0aW9ucy5yb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogW3RoaXMucHJvcHMucmVwb3NpdG9yeS5yZXBvc2l0b3J5QXJuXSxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2NvZGVjb21taXQ6R2V0QnJhbmNoJyxcbiAgICAgICAgJ2NvZGVjb21taXQ6R2V0Q29tbWl0JyxcbiAgICAgICAgJ2NvZGVjb21taXQ6VXBsb2FkQXJjaGl2ZScsXG4gICAgICAgICdjb2RlY29tbWl0OkdldFVwbG9hZEFyY2hpdmVTdGF0dXMnLFxuICAgICAgICAnY29kZWNvbW1pdDpDYW5jZWxVcGxvYWRBcmNoaXZlJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgUmVwb3NpdG9yeU5hbWU6IHRoaXMucHJvcHMucmVwb3NpdG9yeS5yZXBvc2l0b3J5TmFtZSxcbiAgICAgICAgQnJhbmNoTmFtZTogdGhpcy5icmFuY2gsXG4gICAgICAgIFBvbGxGb3JTb3VyY2VDaGFuZ2VzOiB0aGlzLnByb3BzLnRyaWdnZXIgPT09IENvZGVDb21taXRUcmlnZ2VyLlBPTEwsXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGdlbmVyYXRlRXZlbnRJZChzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSk6IHN0cmluZyB7XG4gICAgY29uc3QgYmFzZUlkID0gTmFtZXMubm9kZVVuaXF1ZUlkKHN0YWdlLnBpcGVsaW5lLm5vZGUpO1xuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQodGhpcy5icmFuY2gpKSB7XG4gICAgICBsZXQgY2FuZGlkYXRlID0gJyc7XG4gICAgICBsZXQgY291bnRlciA9IDA7XG4gICAgICBkbyB7XG4gICAgICAgIGNhbmRpZGF0ZSA9IHRoaXMuZXZlbnRJZEZyb21QcmVmaXgoYCR7YmFzZUlkfSR7Y291bnRlcn1gKTtcbiAgICAgICAgY291bnRlciArPSAxO1xuICAgICAgfSB3aGlsZSAodGhpcy5wcm9wcy5yZXBvc2l0b3J5Lm5vZGUudHJ5RmluZENoaWxkKGNhbmRpZGF0ZSkgIT09IHVuZGVmaW5lZCk7XG4gICAgICByZXR1cm4gY2FuZGlkYXRlO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBicmFuY2hJZERpc2FtYmlndWF0b3IgPSB0aGlzLmJyYW5jaCA9PT0gJ21hc3RlcicgPyAnJyA6IGAtJHt0aGlzLmJyYW5jaH0tYDtcbiAgICAgIHJldHVybiB0aGlzLmV2ZW50SWRGcm9tUHJlZml4KGAke2Jhc2VJZH0ke2JyYW5jaElkRGlzYW1iaWd1YXRvcn1gKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGV2ZW50SWRGcm9tUHJlZml4KGV2ZW50SWRQcmVmaXg6IHN0cmluZykge1xuICAgIHJldHVybiBgJHtldmVudElkUHJlZml4fUV2ZW50UnVsZWA7XG4gIH1cbn1cbiJdfQ==