import * as codecommit from '@aws-cdk/aws-codecommit';
import * as codepipeline from '@aws-cdk/aws-codepipeline';
import * as iam from '@aws-cdk/aws-iam';
import { Construct } from '@aws-cdk/core';
import { Action } from '../action';
/**
 * How should the CodeCommit Action detect changes.
 *
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 *
 * @stability stable
 */
export declare enum CodeCommitTrigger {
    /**
     * The Action will never detect changes - the Pipeline it's part of will only begin a run when explicitly started.
     *
     * @stability stable
     */
    NONE = "None",
    /**
     * CodePipeline will poll the repository to detect changes.
     *
     * @stability stable
     */
    POLL = "Poll",
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     *
     * This is the default method of detecting changes.
     *
     * @stability stable
     */
    EVENTS = "Events"
}
/**
 * The CodePipeline variables emitted by the CodeCommit source Action.
 *
 * @stability stable
 */
export interface CodeCommitSourceVariables {
    /**
     * The name of the repository this action points to.
     *
     * @stability stable
     */
    readonly repositoryName: string;
    /**
     * The name of the branch this action tracks.
     *
     * @stability stable
     */
    readonly branchName: string;
    /**
     * The date the currently last commit on the tracked branch was authored, in ISO-8601 format.
     *
     * @stability stable
     */
    readonly authorDate: string;
    /**
     * The date the currently last commit on the tracked branch was committed, in ISO-8601 format.
     *
     * @stability stable
     */
    readonly committerDate: string;
    /**
     * The SHA1 hash of the currently last commit on the tracked branch.
     *
     * @stability stable
     */
    readonly commitId: string;
    /**
     * The message of the currently last commit on the tracked branch.
     *
     * @stability stable
     */
    readonly commitMessage: string;
}
/**
 * Construction properties of the {@link CodeCommitSourceAction CodeCommit source CodePipeline Action}.
 *
 * @stability stable
 */
export interface CodeCommitSourceActionProps extends codepipeline.CommonAwsActionProps {
    /**
     * @stability stable
     */
    readonly output: codepipeline.Artifact;
    /**
     * @default 'master'
     * @stability stable
     */
    readonly branch?: string;
    /**
     * How should CodePipeline detect source changes for this Action.
     *
     * @default CodeCommitTrigger.EVENTS
     * @stability stable
     */
    readonly trigger?: CodeCommitTrigger;
    /**
     * The CodeCommit repository.
     *
     * @stability stable
     */
    readonly repository: codecommit.IRepository;
    /**
     * Role to be used by on commit event rule.
     *
     * Used only when trigger value is CodeCommitTrigger.EVENTS.
     *
     * @default a new role will be created.
     * @stability stable
     */
    readonly eventRole?: iam.IRole;
}
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 *
 * @stability stable
 */
export declare class CodeCommitSourceAction extends Action {
    private readonly branch;
    private readonly props;
    /**
     * @stability stable
     */
    constructor(props: CodeCommitSourceActionProps);
    /**
     * The variables emitted by this action.
     *
     * @stability stable
     */
    get variables(): CodeCommitSourceVariables;
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    protected bound(_scope: Construct, stage: codepipeline.IStage, options: codepipeline.ActionBindOptions): codepipeline.ActionConfig;
    private generateEventId;
    private eventIdFromPrefix;
}
