"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeleteStackAction = exports.CloudFormationCreateUpdateStackAction = exports.CloudFormationCreateReplaceChangeSetAction = exports.CloudFormationExecuteChangeSetAction = void 0;
const cloudformation = require("@aws-cdk/aws-cloudformation");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 *
 * @stability stable
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     *
     * @stability stable
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    /**
     * @stability stable
     */
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${cdk.Names.nodeUniqueId(stage.pipeline.node)}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 *
 * @stability stable
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 *
 * @stability stable
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 *
 * @stability stable
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    /**
     * The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @stability stable
     */
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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