"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const util_1 = require("projen/lib/util");
const generated_typescript_client_project_1 = require("./codegen/generated-typescript-client-project");
const languages_1 = require("./languages");
const typescript_2 = require("./samples/typescript");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "@aws-prototyping-sdk/open-api-gateway";
/**
 * Synthesizes a Typescript Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-ts
 */
class OpenApiGatewayTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                },
            },
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Allow json files to be imported (for importing the parsed spec)
        this.tsconfig?.addInclude(`${this.srcdir}/**/*.json`);
        // Set to private since this either uses workspaces or has file dependencies
        this.package.addField("private", true);
        // Generated project should have a dependency on this project, in order to run the generation scripts
        this.addDeps(OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib");
        // Synthesize the openapi spec early since it's used by the generated typescript client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.srcdir, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        new projen_1.TextFile(this, path.join(this.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Clients",
                "",
                "This directory contains generated client code based on your OpenAPI Specification file (spec.yaml).",
                "",
                "Like other `projen` managed files, this directory should be checked in to source control, but should not be edited manually.",
            ],
            readonly: true,
        });
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // We generate the typescript client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        this.generatedTypescriptClient = new generated_typescript_client_project_1.GeneratedTypescriptClientProject({
            parent: this.hasParent ? options.parent : this,
            defaultReleaseBranch: options.defaultReleaseBranch,
            name: `${this.package.packageName}-typescript`,
            outdir: path.join(generatedCodeDirRelativeToParent, languages_1.ClientLanguage.TYPESCRIPT),
            // Use the parsed spec such that refs are resolved to support multi-file specs
            specPath: spec.parsedSpecPath,
            packageManager: options.packageManager,
        });
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedTypescriptClient.synth();
        const typescriptCodeGenDir = path.relative(this.outdir, this.generatedTypescriptClient.outdir);
        if (this.hasParent) {
            // When we have a parent project, we can add a dependency on the generated typescript client since it's part of
            // the monorepo. Since this project will be synthesized before the parent monorepo, the generated typescript
            // client won't be visible for the first install in this project's post synthesize step, so we use a local
            // workspace as a workaround.
            if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
                this.pnpmWorkspace = new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                    readonly: true,
                    obj: {
                        packages: [typescriptCodeGenDir],
                    },
                });
            }
            else {
                this.package.addField("workspaces", {
                    packages: [typescriptCodeGenDir],
                });
            }
            // Add the dependency
            this.addDeps(this.generatedTypescriptClient.package.packageName);
            // Since the generated typescript client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        else {
            // Add a file dependency on the generated typescript client
            this.addDeps(`${this.generatedTypescriptClient.package.packageName}@file:./${typescriptCodeGenDir}`);
            // Linting runs post synthesize before the typescript client has been built
            this.eslint?.addRules({
                "import/no-unresolved": [
                    2,
                    { ignore: [this.generatedTypescriptClient.package.packageName] },
                ],
            });
        }
        // Additional languages to generate other than typescript which is mandatory
        const extraLanguages = new Set(options.clientLanguages);
        extraLanguages.delete(languages_1.ClientLanguage.TYPESCRIPT);
        // TODO: generate clients in other given languages
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            typescriptClientPackageName: this.generatedTypescriptClient.package.packageName,
            sampleCode: options.sampleCode,
            apiSrcDir: path.join(this.srcdir, this.apiSrcDir),
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
        };
        new projen_1.SampleFile(this, path.join(this.srcdir, "index.ts"), {
            contents: `export * from "./${this.apiSrcDir}";`,
        });
        new projen_1.SampleDir(this, path.join(this.srcdir, this.apiSrcDir), {
            files: typescript_2.getTypescriptSampleSource(sampleOptions),
        });
        new projen_1.SampleDir(this, this.testdir, {
            files: typescript_2.getTypescriptSampleTests(sampleOptions),
        });
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // When no parent is passed, link the generated client as a prebuild step to ensure the latest built generated
        // client is reflected in this package's node modules.
        // Note that it's up to the user to manage building the generated client first.
        !this.hasParent && this.executeLinkNativeClientCommands();
        super.postSynthesize();
    }
    /**
     * Executes commands to link the native (ie typescript) client such that updates to the generated client are reflected
     * in this project's node_modules (when not managed by a monorepo)
     * @private
     */
    executeLinkNativeClientCommands() {
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.NPM:
            case javascript_1.NodePackageManager.YARN:
                util_1.exec(`${this.package.packageManager} link`, {
                    cwd: this.generatedTypescriptClient.outdir,
                });
                util_1.exec(`${this.package.packageManager} link ${this.generatedTypescriptClient.package.packageName}`, {
                    cwd: this.outdir,
                });
                break;
            case javascript_1.NodePackageManager.PNPM:
                util_1.exec(`${this.package.packageManager} link ./${path.relative(this.outdir, this.generatedTypescriptClient.outdir)}`, {
                    cwd: this.outdir,
                });
                break;
            default:
                throw new Error(`Unsupported package manager ${this.package.packageManager}`);
        }
    }
}
exports.OpenApiGatewayTsProject = OpenApiGatewayTsProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayTsProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayTsProject", version: "0.2.0" };
//# sourceMappingURL=data:application/json;base64,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