#
#     Copyright 2021 Joël Larose
#
#     Licensed under the Apache License, Version 2.0 (the "License");
#     you may not use this file except in compliance with the License.
#     You may obtain a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#     Unless required by applicable law or agreed to in writing, software
#     distributed under the License is distributed on an "AS IS" BASIS,
#     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#     See the License for the specific language governing permissions and
#     limitations under the License.
#
#
"""Main module for Python Parallel Hierarchy Factory."""


from __future__ import annotations
from typing import Generic, TypeVar, Type, Any, ClassVar, Dict, List, Union, cast

SourceBase = TypeVar('SourceBase')
SourceBase.__doc__ = """This type variable designates the root of the hierarchy for the source
classes which will be paralleled."""

ParaBase = TypeVar('ParaBase')
ParaBase.__doc__ = """This type variable designates the root of the parallel hierarchy."""

class ParallelFactory(Generic[SourceBase, ParaBase]):
    """
    Base class for factories a parallel set of classes for the Entity class tree.

    Usage
    =====
    Subclass `ParallelFactory` providing types for the SourceBase and ParaBase generic parameters,
    and either prefix or suffix (or both) as named arguments.

    Example
    =======
    ```
    class Foos(ParallelFactory[RootClass, Foo], suffix='Foo'):
        pass
    ```

    where `RootClass` is the root of the original class tree, and `Foo` is the root of the class
    tree paralleling the original tree

    `Foos(SubClass)` will look for a class called `SubClassFoo`, and if it doesn't exist then
    create it.  Returns the result of the search or creation.

    Class Parameters
    ================
    The following arguments are allowed during class definition of a subclass of `ParallelFactory`:
    SourceBase
        Base class for the set of classes which will be paralleled with this factory.  (Required)
    ParaBase
        Base class for the parallel set of classes generated by this factory class.  (Required)
    prefix
        Common prefix for all subclasses of para_base. (Default: '')
    suffix
        Common suffix for all subclasses of para_base. (Default: '')

    .. note::
        At least one of prefix or suffix must be provided as a non-empty string.  These are used
        both for retrieving and creating subclasses.
    """

    # pylint: disable=invalid-name
    _parallel_classes: ClassVar[Dict[str, Type[ParaBase]]] = dict()
    source_base_class: ClassVar[Type[SourceBase]]
    parallel_base_class: ClassVar[Type[ParaBase]]
    parallel_prefix: ClassVar[str]
    parallel_suffix: ClassVar[str]

    @classmethod
    def __init_subclass__(cls, prefix: str = "", suffix: str = "", **_kwargs) -> None:
        """
        Defines properties of the factory class.

        :param prefix: Prefix used to lookup and, if needed, create a class.
        :param suffix: Suffix used to lookup and, if needed, create a class.
        :param kwargs:
        """
        super().__init_subclass__()

        # Get Generic parameter values
        gen_args = cls.__orig_bases__[0].__args__  # type: ignore # pylint: disable=no-member
        print(f'{cls.__name__}.gen_args = {gen_args}')
        source_base = gen_args[0]
        para_base = gen_args[1]

        err_msgs: List[str] = []

        # Parameter defense
        if source_base is SourceBase:       # type: ignore[misc]
            # Means no argument was provided for the SourceBase type variable
            err_msgs.append("SourceBase must be provided")
        if para_base is ParaBase:           # type: ignore[misc]
            # Means no argument was provided for the ParaBase type variable
            err_msgs.append("ParaBase must be provided")
        if not isinstance(source_base, type):
            err_msgs.append("SourceBase must be a type")
        if not isinstance(para_base, type):
            err_msgs.append("ParaBase must be a type")
        if err_msgs:
            raise TypeError(err_msgs)

        if prefix is None:
            prefix = ""
        if suffix is None:
            suffix = ""
        if prefix + suffix == "":
            raise AttributeError("At least one of 'prefix' or 'suffix' must be specified "
                                 "with a non-empty string.")
        cls.source_base_class = source_base
        cls.parallel_base_class = para_base
        cls.parallel_prefix = prefix
        cls.parallel_suffix = suffix

    @classmethod
    def register(cls, parallel_class: Type[ParaBase]) -> None:
        """
        Adds the `parallel_class` to inventory of known classes.

        The `register` method gets called by `build_parallel_class`, which gets called when a
        lookup fails.  Custom parallel classes must be registered explicitly to work properly with
        the factory's lookup.  Here are some options for registering the custom parallel class:

        * Call `register` in a metaclass for `ParaBase`
        * Call `register` in `__init_subclass__` defined in `ParaBase`
        * Call `register` immediately after defining each custom class.

        :param parallel_class: The class to be added in the parallel hierarchy.
        """
        if not issubclass(parallel_class, cls.parallel_base_class):
            raise TypeError("'parallel_class' must be a subclass of "
                            f"{str(cls.parallel_base_class)}, got {str(parallel_class)}.")
        cls._parallel_classes[parallel_class.__name__] = parallel_class

    def __new__(cls, source_class: Type[SourceBase]) -> Type[ParaBase]:    # type: ignore[misc]
        """
        Alias for `get`.

        .. warning::
            This purposely does not create an instance of the factory class.
        """
        return cls.get(source_class)

    @classmethod
    def get(cls, source_class: Type[SourceBase]) -> Type[ParaBase]:
        """
        Looks up registered parallel classes by name using [pre/suf]fix and source_class name.
        If found, return it, otherwise, generate it.

        :param source_class: The source class you wish to parallel.
        :return: The parallel counterpart of `source_class`
        """
        if source_class is None:
            raise ValueError('source_class required')
        if not issubclass(source_class, cls.source_base_class):
            raise TypeError(
                    "'source_class' must be a subclass of "
                    f"{str(cls.source_base_class)}, got {str(source_class)}")

        p_name = cls._name_fix(source_class.__name__)
        if p_name in cls._parallel_classes:
            parallel_class = cls._parallel_classes[p_name]
        else:
            parallel_class = cls.build_parallel_class(source_class, p_name)
        return parallel_class

    @classmethod
    def build_parallel_class(cls, source_class: Type[SourceBase], p_name: str,
                             **extra_attrs: Any) -> Type[ParaBase]:
        """
        Builds the parallel class, ensuring that it's ancestors have also been created.

        Preserves the parallel hierarchy down to the

        `source_class` and any attributes passed through `extra_attr` will be added to the
        newly built class.  These can be used by a metaclass for the class passed as `ParaBase`.

        :param source_class: Source class for which the parallel will be built.
        :param p_name: Name of the new parallel class.
        :param extra_attrs: Additional attributes that will be added to the new class.
        :return: Newly created parallel counterpart of `source_class`
        """
        parallel_attrs = {'source_class': source_class, **extra_attrs}

        super_p_class: List[Type[ParaBase]] = []
        # generated class assumes a parallel hierarchy
        if source_class == cls.source_base_class:  # recursion stopper
            super_p_class.append(cls.parallel_base_class)
        elif issubclass(source_class, cls.source_base_class):
            for base in source_class.__bases__:
                if issubclass(base, cls.source_base_class):
                    super_p_class.append(cls.get(base))  # Recursive step
                # else:
                #     print("Proof I was here")  # Just for coverage assessment
        else:
            raise TypeError()

        # Allows for metaclasses other than `type`
        meta: Type[Type[ParaBase]] = type(cls.parallel_base_class)
        p_class = cast(Type[ParaBase],
                       meta(p_name, tuple(super_p_class), parallel_attrs))  # type: ignore[misc]
        cls._parallel_classes[p_name] = p_class

        return p_class

    @classmethod
    def _name_fix(cls, cls_name: str) -> str:
        return cls.parallel_prefix + cls_name + cls.parallel_suffix

    @classmethod
    def has(cls, item: Union[str, type]) -> bool:
        """
        Checks to see if a class has been registered.

        :param item: The class, or name of a class, to test for registration.
        """
        pcs = cls._parallel_classes

        if isinstance(item, str):
            return item in pcs or cls._name_fix(item) in pcs
        if isinstance(item, type):
            if issubclass(item, cls.source_base_class):
                return cls._name_fix(item.__name__) in pcs
            if issubclass(item, cls.parallel_base_class):
                # Test both existence of the key and that the class matches
                return item.__name__ in pcs and item == pcs[item.__name__]

        # Not a str, or not a type, or not the right type
        return False
