"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const util_1 = require("projen/lib/util");
const generated_typescript_client_project_1 = require("./codegen/generated-typescript-client-project");
const languages_1 = require("./languages");
const typescript_2 = require("./samples/typescript");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "@aws-prototyping-sdk/open-api-gateway";
/**
 * Synthesizes a Typescript Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-ts
 */
class OpenApiGatewayTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            // Default src dir to 'api' to allow for more readable imports, eg `import { SampleApi } from 'my-generated-api/api'`.
            srcdir: options.srcdir || "api",
            tsconfig: {
                compilerOptions: {
                    // Root dir needs to include srcdir and spec
                    rootDir: ".",
                    lib: ["dom", "es2019"],
                },
            },
        });
        this.specDir = options.specDir ?? "spec";
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        // Allow spec to be imported, using both the source and spec directories as project roots.
        this.tsconfig?.addInclude(`${this.specDir}/**/*.json`);
        this.package.addField("main", path.join(this.libdir, this.srcdir, "index.js"));
        this.package.addField("types", path.join(this.libdir, this.srcdir, "index.d.ts"));
        // Set to private since this either uses workspaces or has file dependencies
        this.package.addField("private", true);
        // Generated project should have a dependency on this project, in order to run the generation scripts
        this.addDeps(OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib");
        // Synthesize the openapi spec early since it's used by the generated typescript client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: this.specDir,
            specFileName: options.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        new projen_1.TextFile(this, path.join(this.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Clients",
                "",
                "This directory contains generated client code based on your OpenAPI Specification file (spec.yaml).",
                "",
                "Like other `projen` managed files, this directory should be checked in to source control, but should not be edited manually.",
            ],
            readonly: true,
        });
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // We generate the typescript client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        this.generatedTypescriptClient = new generated_typescript_client_project_1.GeneratedTypescriptClientProject({
            parent: this.hasParent ? options.parent : this,
            defaultReleaseBranch: options.defaultReleaseBranch,
            name: `${this.package.packageName}-typescript`,
            outdir: path.join(generatedCodeDirRelativeToParent, languages_1.ClientLanguage.TYPESCRIPT),
            // Use the parsed spec such that refs are resolved to support multi-file specs
            specPath: spec.parsedSpecPath,
            packageManager: options.packageManager,
        });
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedTypescriptClient.synth();
        const typescriptCodeGenDir = path.relative(this.outdir, this.generatedTypescriptClient.outdir);
        if (this.hasParent) {
            // When we have a parent project, we can add a dependency on the generated typescript client since it's part of
            // the monorepo. Since this project will be synthesized before the parent monorepo, the generated typescript
            // client won't be visible for the first install in this project's post synthesize step, so we use a local
            // workspace as a workaround.
            if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
                this.pnpmWorkspace = new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                    readonly: true,
                    obj: {
                        packages: [typescriptCodeGenDir],
                    },
                });
            }
            else {
                this.package.addField("workspaces", {
                    packages: [typescriptCodeGenDir],
                });
            }
            // Add the dependency
            this.addDeps(this.generatedTypescriptClient.package.packageName);
            // Since the generated typescript client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        else {
            // Add a file dependency on the generated typescript client
            this.addDeps(`${this.generatedTypescriptClient.package.packageName}@file:./${typescriptCodeGenDir}`);
            // Linting runs post synthesize before the typescript client has been built
            this.eslint?.addRules({
                "import/no-unresolved": [
                    2,
                    { ignore: [this.generatedTypescriptClient.package.packageName] },
                ],
            });
        }
        // Additional languages to generate other than typescript which is mandatory
        const extraLanguages = new Set(options.clientLanguages);
        extraLanguages.delete(languages_1.ClientLanguage.TYPESCRIPT);
        // TODO: generate clients in other given languages
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            typescriptClientPackageName: this.generatedTypescriptClient.package.packageName,
            sampleCode: options.sampleCode,
            srcdir: this.srcdir,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
        };
        new projen_1.SampleDir(this, this.srcdir, {
            files: typescript_2.getTypescriptSampleSource(sampleOptions),
        });
        new projen_1.SampleDir(this, this.testdir, {
            files: typescript_2.getTypescriptSampleTests(sampleOptions),
        });
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // When no parent is passed, link the generated client as a prebuild step to ensure the latest built generated
        // client is reflected in this package's node modules.
        // Note that it's up to the user to manage building the generated client first.
        !this.hasParent && this.executeLinkNativeClientCommands();
        super.postSynthesize();
    }
    /**
     * Executes commands to link the native (ie typescript) client such that updates to the generated client are reflected
     * in this project's node_modules (when not managed by a monorepo)
     * @private
     */
    executeLinkNativeClientCommands() {
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.NPM:
            case javascript_1.NodePackageManager.YARN:
                util_1.exec(`${this.package.packageManager} link`, {
                    cwd: this.generatedTypescriptClient.outdir,
                });
                util_1.exec(`${this.package.packageManager} link ${this.generatedTypescriptClient.package.packageName}`, {
                    cwd: this.outdir,
                });
                break;
            case javascript_1.NodePackageManager.PNPM:
                util_1.exec(`${this.package.packageManager} link ./${path.relative(this.outdir, this.generatedTypescriptClient.outdir)}`, {
                    cwd: this.outdir,
                });
                break;
            default:
                throw new Error(`Unsupported package manager ${this.package.packageManager}`);
        }
    }
}
exports.OpenApiGatewayTsProject = OpenApiGatewayTsProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayTsProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayTsProject", version: "0.0.96" };
//# sourceMappingURL=data:application/json;base64,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