mod client;
mod enums;
mod mods;
mod path;

use anyhow::Result as AnyResult;
use client::ApiDetailedDescription;
use mods::Mods;
use serde_yaml::from_reader as yaml_from_reader;
use std::{
    collections::VecDeque,
    env::current_dir,
    ffi::{OsStr, OsString},
    fs::{create_dir_all, remove_dir_all, write, File},
    path::{Component, Path, PathBuf},
};
use walkdir::{DirEntry, WalkDir};

fn main() -> AnyResult<()> {
    env_logger::init();

    generate_python_modules()
}

fn generate_python_modules() -> AnyResult<()> {
    let mut mods = Mods::default();
    let current_dir = current_dir()?;
    let api_spec_dir = current_dir.join("api-specs");
    let apis_generated_dir = current_dir.join("src").join("apis");
    remove_dir_all(&apis_generated_dir).ok();
    create_dir_all(&apis_generated_dir)?;

    for read_result in WalkDir::new(&api_spec_dir)
        .into_iter()
        .filter_entry(|e| !is_hidden(e))
    {
        let entry = read_result?;
        let namespace = extract_namespace(&entry, &api_spec_dir)?;
        let file_name = Path::new(entry.file_name());
        if matches!(
            Path::new(entry.file_name())
                .extension()
                .and_then(OsStr::to_str),
            Some("yml" | "yaml")
        ) {
            write_to_python_mod(
                file_name.with_extension(""),
                namespace,
                entry.path(),
                &apis_generated_dir,
                &mut mods,
            )?;
        }
    }
    mods.write_to_python_mod("apis", &apis_generated_dir)?;
    mods.write_sphinx_index("apis", &apis_generated_dir)?;

    return Ok(());

    fn is_hidden(entry: &DirEntry) -> bool {
        entry
            .file_name()
            .to_str()
            .map(|s| s.starts_with('.'))
            .unwrap_or(false)
    }

    fn extract_namespace(entry: &DirEntry, prefix: &Path) -> AnyResult<Vec<OsString>> {
        let namespace = entry
            .path()
            .strip_prefix(prefix)?
            .parent()
            .map(|dir_path| {
                dir_path
                    .components()
                    .map(|component| {
                        if let Component::Normal(component) = component {
                            component.to_owned()
                        } else {
                            unreachable!()
                        }
                    })
                    .collect()
            })
            .unwrap_or_default();
        Ok(namespace)
    }

    fn write_to_python_mod(
        base_name: PathBuf,
        namespace: Vec<OsString>,
        api_spec_path: &Path,
        apis_generated_dir: &Path,
        mods: &mut Mods,
    ) -> AnyResult<()> {
        let output_file_path = namespace
            .iter()
            .fold(apis_generated_dir.to_owned(), |path, segment| {
                path.join(segment)
            })
            .join(Path::new(&base_name).with_extension("rs"));
        if let Some(output_dir_path) = output_file_path.parent() {
            create_dir_all(output_dir_path)?;
        }
        let api_detailed_spec: ApiDetailedDescription =
            yaml_from_reader(File::open(api_spec_path)?)?;
        let auto_generated_code =
            "// THIS FILE IS GENERATED BY api-generator, DO NOT EDIT DIRECTLY!\n//\n".to_owned()
                + &api_detailed_spec
                    .to_python_token_stream(base_name.to_str().unwrap())
                    .to_string();
        write(&output_file_path, auto_generated_code.as_bytes())?;

        mods.add(base_name.into_os_string(), VecDeque::from_iter(namespace));

        Ok(())
    }
}
