"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const events = require("aws-cdk-lib/aws-events");
const ecs = require("aws-cdk-lib/aws-ecs");
const clusterName = "custom-cluster-name";
const containerName = "custom-container-name";
const serviceName = "custom-service-name";
const familyName = "family-name";
const testCidr = "172.0.0.0/16";
test('Check for construct properties', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = createFargateConstructWithNewResources(stack, publicApi);
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.eventBus).toBeDefined();
});
test('Check for new service', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".events"
                ]
            ]
        }
    });
});
test('Check for an existing service', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const existingVpc = defaults.getTestVpc(stack);
    const [testService, testContainer] = defaults.CreateFargateService(stack, 'test', existingVpc, { clusterName }, defaults.fakeEcrRepoArn, undefined, { family: familyName }, { containerName }, { serviceName });
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: testService,
        existingContainerDefinitionObject: testContainer,
        existingVpc,
    });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".events"
                ]
            ]
        }
    });
    expect(stack).toCountResources("AWS::ECS::Service", 1);
});
test('Check for IAM put events policy for created bus', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "events:PutEvents",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testconstructcustomnameA27657F0",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
});
test('Check for IAM put events policy for default event bus', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
    });
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "events:PutEvents",
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":events:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":event-bus/default"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
});
test('Check for public/private VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    expect(stack).toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::Events::EventBus', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Check for isolated VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    expect(stack).not.toHaveResourceLike('AWS::EC2::InternetGateway', {});
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toCountResources('AWS::Events::EventBus', 1);
    expect(stack).toCountResources('AWS::ECS::Service', 1);
});
test('Check for an existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingVpc,
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16"
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
});
test('Check for custom ARN resource', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const customEnvName = "TEST_CUSTOM_ARN";
    new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        eventBusEnvironmentVariableName: customEnvName
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Family: familyName,
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customEnvName,
                        Value: "default"
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
});
test('Check for an existing event bus', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const customName = 'MyCustomEventBus';
    const existingEventBus = new events.EventBus(stack, 'bus', {
        eventBusName: customName
    });
    const construct = new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingEventBusInterface: existingEventBus
    });
    expect(construct.eventBus).toBeDefined();
    expect(stack).toHaveResource("AWS::Events::EventBus", {
        Name: customName
    });
});
test('Check for custom event bus props', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const eventBusName = "custom-name";
    createFargateConstructWithNewResources(stack, publicApi);
    expect(stack).toHaveResourceLike("AWS::Events::EventBus", {
        Name: eventBusName
    });
});
function createFargateConstructWithNewResources(stack, publicApi) {
    return new lib_1.FargateToEventbridge(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { cidr: testCidr },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        eventBusProps: {
            eventBusName: 'custom-name'
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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