"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.concatMethodAndPath = void 0;
const constants_1 = require("./constants");
/**
 * Serialise a method and path into a single string
 */
exports.concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
// Add to methods to ensure no auth is added
const NO_AUTH_SPEC_SNIPPET = {
    security: [],
    "x-amazon-apigateway-auth": {
        type: "NONE",
    },
};
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 */
const applyMethodAuthorizer = (methodAuthorizer) => {
    if (methodAuthorizer) {
        if (methodAuthorizer.authorizerId === constants_1.DefaultAuthorizerIds.NONE) {
            return NO_AUTH_SPEC_SNIPPET;
        }
        else {
            return {
                security: [
                    {
                        [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                    },
                ],
            };
        }
    }
    return {};
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { methodAuthorizer, integration } = integrations[operationName];
    validateAuthorizerReference(methodAuthorizer, operation.security, operationName);
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": integration,
        ...applyMethodAuthorizer(methodAuthorizer),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (constants_1.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [constants_1.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
            // No auth for CORS options requests
            ...NO_AUTH_SPEC_SNIPPET,
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    const supportedHttpMethods = new Set(Object.values(constants_1.HttpMethods));
    const unsupportedMethodsInSpec = Object.keys(pathItem).filter((method) => !supportedHttpMethods.has(method));
    if (unsupportedMethodsInSpec.length > 0) {
        throw new Error(`Path ${path} contains unsupported method${unsupportedMethodsInSpec.length > 1 ? "s" : ""} ${unsupportedMethodsInSpec.join(", ")}. Supported methods are ${Object.values(constants_1.HttpMethods).join(", ")}.`);
    }
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(constants_1.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return whether the given OpenAPI object is a reference object
 */
const isRef = (obj) => "$ref" in obj;
/**
 * Validate the construct security schemes against the security schemes in the original spec.
 * Construct-defined authorizers always override those in the spec if they have the same ID, however we validate that
 * we are not overriding an authorizer of a different type to avoid mistakes/mismatches between the spec and the
 * construct.
 * @param constructSecuritySchemes security schemes generated from the construct authorizers
 * @param existingSpecSecuritySchemes security schemes already defined in the spec
 */
const validateSecuritySchemes = (constructSecuritySchemes, existingSpecSecuritySchemes) => {
    if (existingSpecSecuritySchemes) {
        const constructSecuritySchemeIds = new Set(Object.keys(constructSecuritySchemes));
        const existingSecuritySchemeIds = new Set(Object.keys(existingSpecSecuritySchemes));
        const overlappingSecuritySchemeIds = [...constructSecuritySchemeIds].filter((id) => existingSecuritySchemeIds.has(id));
        // Any overlapping security schemes (defined in both the spec (or source smithy model) and the construct) must be of the same type.
        // The one defined in the construct will take precedence since a custom/cognito authorizer can have a resolved arn in the construct,
        // and we allow usage in the model as a forward definition with blank arn.
        overlappingSecuritySchemeIds.forEach((schemeId) => {
            if (!isRef(existingSpecSecuritySchemes[schemeId])) {
                const existingScheme = existingSpecSecuritySchemes[schemeId];
                if (constructSecuritySchemes[schemeId].type !== existingScheme.type) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructSecuritySchemes[schemeId].type} in construct but ${existingScheme.type} in OpenAPI spec or Smithy model.`);
                }
                const constructApiGatewayAuthType = constructSecuritySchemes[schemeId]["x-amazon-apigateway-authtype"];
                const existingApiGatewayAuthType = existingScheme["x-amazon-apigateway-authtype"];
                if (constructApiGatewayAuthType !== existingApiGatewayAuthType) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructApiGatewayAuthType} in construct but ${existingApiGatewayAuthType} in OpenAPI spec or Smithy model.`);
                }
            }
            else {
                throw new Error(`Security scheme with id ${schemeId} is a reference in the OpenAPI spec or Smithy model which is not supported.`);
            }
        });
    }
};
/**
 * Validate the given authorizer reference (either default or at an operation level) defined in the construct against
 * those already in the spec.
 * @param constructAuthorizer the authorizer defined in the construct
 * @param existingSpecAuthorizers the authorizers already defined in the spec
 * @param operation the operation we are validating (for clearer error messages)
 */
const validateAuthorizerReference = (constructAuthorizer, existingSpecAuthorizers, operation = "Default") => {
    // Only need to validate if defined in both - if just one we'll use that.
    if (constructAuthorizer && existingSpecAuthorizers) {
        const mergedSpecAuthorizers = Object.fromEntries(existingSpecAuthorizers.flatMap((securityRequirement) => Object.keys(securityRequirement).map((id) => [
            id,
            securityRequirement[id],
        ])));
        const specAuthorizerIds = Object.keys(mergedSpecAuthorizers);
        if (specAuthorizerIds.length > 1) {
            // Spec defined multiple authorizers but the construct can only specify one
            throw new Error(`${operation} authorizers ${specAuthorizerIds
                .sort()
                .join(", ")} defined in the OpenAPI Spec or Smithy Model would be overridden by single construct authorizer ${constructAuthorizer.authorizerId}`);
        }
        else if (specAuthorizerIds.length === 1) {
            // Single authorizer - check that they have the same id
            if (specAuthorizerIds[0] !== constructAuthorizer.authorizerId) {
                throw new Error(`${operation} authorizer ${specAuthorizerIds[0]} defined in the OpenAPI Spec or Smithy Model would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
            }
            // Check that there are no differing scopes between the construct and spec
            const specScopes = new Set(mergedSpecAuthorizers[specAuthorizerIds[0]]);
            const constructScopes = new Set(constructAuthorizer.authorizationScopes);
            const differingScopes = [
                ...[...specScopes].filter((scope) => !constructScopes.has(scope)),
                ...[...constructScopes].filter((scope) => !specScopes.has(scope)),
            ];
            if (differingScopes.length > 0) {
                throw new Error(`${operation} authorizer scopes ${[...specScopes].join(", ")} defined in the OpenAPI Spec or Smithy Model differ from those in the construct (${[
                    ...constructScopes,
                ].join(", ")})`);
            }
        }
        else if (constructAuthorizer.authorizerId !== constants_1.DefaultAuthorizerIds.NONE) {
            // "security" section of spec is [] which means no auth, but the authorizer in the construct is not the "none" authorizer.
            throw new Error(`${operation} explicitly defines no auth in the OpenAPI Spec or Smithy Model which would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
        }
    }
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
exports.prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        exports.concatMethodAndPath(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[exports.concatMethodAndPath(methodAndPath)];
    validateSecuritySchemes(options.securitySchemes, spec.components?.securitySchemes);
    validateAuthorizerReference(options.defaultAuthorizerReference, spec.security);
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, options, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: {
                // Apply any security schemes that already exist in the spec
                ...spec.components?.securitySchemes,
                // Construct security schemes override any in the spec with the same id
                ...options.securitySchemes,
            },
        },
        // Apply the default authorizer at the top level
        ...(options.defaultAuthorizerReference
            ? applyMethodAuthorizer(options.defaultAuthorizerReference)
            : {}),
    };
};
//# sourceMappingURL=data:application/json;base64,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