# AUTOGENERATED! DO NOT EDIT! File to edit: ../03_recurse.ipynb.

# %% auto 0
__all__ = ['make_args_hashable', 'RecursionVisualizer', 'on_colab']

# %% ../03_recurse.ipynb 3
from typing import List, Union

from .node import Node 
from .graph import get_graph
from .animate import animate
import recursion_visualizer.animate

# %% ../03_recurse.ipynb 5
def make_args_hashable(args):
    hashable_args = []
    for arg in args:
        if isinstance(arg, list):
            hashable_args.append(tuple(arg))
        elif isinstance(arg, set):
            hashable_args.append(frozenset(arg))
        elif isinstance(arg, dict):
            raise ValueError("Dictionaries are not hashable.")
        else:
            hashable_args.append(arg)
    return tuple(hashable_args)

# %% ../03_recurse.ipynb 6
class RecursionVisualizer:
  """A class that provides a decorator for visualizing recursion trees and caching results."""

  def __init__(self,
               verbose: bool = False, # if true, print all nodes
               animate: bool = True, # if true, create an animation of the recursion tree
               save: bool = False, # if true, save the animation to a html file
               path: str ='', # path to save the animation to
               display_args: Union[bool, List] = True # If True, display all arguments of the recursive function in each node; if list, only display arguments whose indices are in this list
               ): 
    
    self.verbose = verbose
    self.animate = animate
    self.save = save
    self.path = path
    self.display_args = display_args
    self._reset()

  def _reset(self):
    """
    self.nodes = preorder traversal of nodes
    self.history = element i was discovered or finished at time i
    self.pos = position of vertices in animate
    """
    self.nodes, self.node_to_edge_labels, self.history = {}, {}, []
    self.id, self.time, self.depth = 0, 0, 0
    self.cache = {}
    self.func_name = ''

  def _animate(self, history, nodes, node_to_edge_labels, func_name, display_args):
    
    DG, edge_to_label = get_graph(history, nodes, node_to_edge_labels)
    fig = animate(history, nodes, func_name, DG, edge_to_label, display_args)
    fig.show()



    # # create recursion tree animation
    # fig = animate(history, nodes, func_name)
    # fig.show()

    # # save figure
    # if self.save:
    #   if self.path == '':
    #     input = ','.join(list(map(str, nodes[0]['input'])))
    #     self.path = './{}_{}.html'.format(func_name, input)
    #   fig.write_html(self.path)

  def __call__(self, 
               func: callable # function to be visualized or cached via decorator
               ):
    """A custom `__call__` function records the id, function input, function output, depth, discovery time, 
    and finish time in a node each time the function is called. After all function calls are made, `__call__`
    will animate the recursion tree. This is the main workhorse of the `RecursionVisualizer` class.
    
    At a high-level, the `__call__` function looks something like:
    
    ```
    def __call__(self, func):
      def memoized_func(*args, **kwargs):
        # record discovery time, function input, and depth
        node.discovery = time
        node.input = args
        node.depth = depth
        
        # if node not in cache, compute and cache result
        if node not in self.cache:
          self.cache[args] = func(*args, **kwargs)
          
        # record finish time and function output
        node.output = self.cache[args]
        node.finish = time
        
        if depth == 0:
          animate()
        
      return memoized_func
    ```
    """

    def memoized_func(*args, **kwargs):
      if self.depth == 0:
        self._reset()
        
      # make args hashable and display-able
      args = make_args_hashable(args)

      # record node's depth, discovery time, and input arguments
      id_ = len(self.nodes)
      node = Node(id_=id_, input=args, depth=self.depth, discovered=self.time)
      self.history.append(node.id_)
      self.nodes[node.id_] = node
      self.time += 1

      # update depth and call the function `func`
      self.depth += 1
      # if args not in self.cache:
      self.cache[args] = func(*args, **kwargs)
      self.depth -= 1

      # record node's output, finish time, history, and edge_label
      self.nodes[id_].output = self.cache[args]
      self.nodes[id_].finish = self.time
      
      self.node_to_edge_labels[id_] = kwargs['edge_label'] if kwargs and 'edge_label' in kwargs else ''
      self.history.append(node.id_)
      self.time += 1

      if self.verbose:
        print(node)

      # animate after done traversing through the entire tree
      if self.animate and self.depth == 0:
        fig = self._animate(self.history, self.nodes, self.node_to_edge_labels, func.__name__, self.display_args)
        return fig, self.cache[args]

      return self.cache[args]
    return memoized_func

# %% ../03_recurse.ipynb 7
import plotly.io as pio
from IPython.display import display, HTML
from IPython import get_ipython

# %% ../03_recurse.ipynb 8
def on_colab():   # cf https://stackoverflow.com/questions/53581278/test-if-notebook-is-running-on-google-colab
    """Returns true if code is being executed on Colab, false otherwise"""
    try:
        return 'google.colab' in str(get_ipython())
    except NameError:    # no get_ipython, so definitely not on Colab
        return False 

if not on_colab():  # Nick Burrus' code for normal-Juptyer use with plotly:
    pio.renderers.default = 'notebook_connected'
    js = '<script src="https://cdnjs.cloudflare.com/ajax/libs/require.js/2.3.6/require.min.js" integrity="sha512-c3Nl8+7g4LMSTdrm621y7kf9v3SDPnhxLNhcjFJbKECVnmZHTdo+IRO05sNLTH/D3vA6u1X32ehoLC7WFVdheg==" crossorigin="anonymous"></script>'
    display(HTML(js))
