"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeDeployEcsDeployAction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * @stability stable
 */
class CodeDeployEcsDeployAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        const inputs = [];
        inputs.push(determineTaskDefinitionArtifact(props));
        inputs.push(determineAppSpecArtifact(props));
        if (props.containerImageInputs) {
            if (props.containerImageInputs.length > 4) {
                throw new Error(`Action cannot have more than 4 container image inputs, got: ${props.containerImageInputs.length}`);
            }
            for (const imageInput of props.containerImageInputs) {
                inputs.push(imageInput.input);
            }
        }
        super({
            ...props,
            resource: props.deploymentGroup,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'CodeDeployToECS',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 0 },
            inputs,
        });
        this.actionProps = props;
    }
    /**
     * This is a renamed version of the {@link IAction.bind} method.
     *
     * @stability stable
     */
    bound(_scope, _stage, options) {
        // permissions, based on:
        // https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.application.applicationArn],
            actions: ['codedeploy:GetApplication', 'codedeploy:GetApplicationRevision', 'codedeploy:RegisterApplicationRevision'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentGroupArn],
            actions: ['codedeploy:CreateDeployment', 'codedeploy:GetDeployment'],
        }));
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.actionProps.deploymentGroup.deploymentConfig.deploymentConfigArn],
            actions: ['codedeploy:GetDeploymentConfig'],
        }));
        // Allow action to register the task definition template file with ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['ecs:RegisterTaskDefinition'],
        }));
        // Allow passing any roles specified in the task definition template file to ECS
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: ['*'],
            conditions: {
                StringEqualsIfExists: {
                    'iam:PassedToService': [
                        'ecs-tasks.amazonaws.com',
                    ],
                },
            },
        }));
        // the Action's Role needs to read from the Bucket to get artifacts
        options.bucket.grantRead(options.role);
        const taskDefinitionTemplateArtifact = determineTaskDefinitionArtifact(this.actionProps);
        const appSpecTemplateArtifact = determineAppSpecArtifact(this.actionProps);
        const actionConfig = {
            configuration: {
                ApplicationName: this.actionProps.deploymentGroup.application.applicationName,
                DeploymentGroupName: this.actionProps.deploymentGroup.deploymentGroupName,
                TaskDefinitionTemplateArtifact: core_1.Lazy.string({ produce: () => taskDefinitionTemplateArtifact.artifactName }),
                TaskDefinitionTemplatePath: this.actionProps.taskDefinitionTemplateFile
                    ? this.actionProps.taskDefinitionTemplateFile.fileName
                    : 'taskdef.json',
                AppSpecTemplateArtifact: core_1.Lazy.string({ produce: () => appSpecTemplateArtifact.artifactName }),
                AppSpecTemplatePath: this.actionProps.appSpecTemplateFile
                    ? this.actionProps.appSpecTemplateFile.fileName
                    : 'appspec.yaml',
            },
        };
        if (this.actionProps.containerImageInputs) {
            for (let i = 1; i <= this.actionProps.containerImageInputs.length; i++) {
                const imageInput = this.actionProps.containerImageInputs[i - 1];
                actionConfig.configuration[`Image${i}ArtifactName`] = core_1.Lazy.string({ produce: () => imageInput.input.artifactName });
                actionConfig.configuration[`Image${i}ContainerName`] = imageInput.taskDefinitionPlaceholder
                    ? imageInput.taskDefinitionPlaceholder
                    : 'IMAGE';
            }
        }
        return actionConfig;
    }
}
exports.CodeDeployEcsDeployAction = CodeDeployEcsDeployAction;
_a = JSII_RTTI_SYMBOL_1;
CodeDeployEcsDeployAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.CodeDeployEcsDeployAction", version: "1.109.0" };
function determineTaskDefinitionArtifact(props) {
    if (props.taskDefinitionTemplateFile && props.taskDefinitionTemplateInput) {
        throw new Error("Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.taskDefinitionTemplateFile) {
        return props.taskDefinitionTemplateFile.artifact;
    }
    if (props.taskDefinitionTemplateInput) {
        return props.taskDefinitionTemplateInput;
    }
    throw new Error("Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action");
}
function determineAppSpecArtifact(props) {
    if (props.appSpecTemplateFile && props.appSpecTemplateInput) {
        throw new Error("Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action");
    }
    if (props.appSpecTemplateFile) {
        return props.appSpecTemplateFile.artifact;
    }
    if (props.appSpecTemplateInput) {
        return props.appSpecTemplateInput;
    }
    throw new Error("Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action");
}
//# sourceMappingURL=data:application/json;base64,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