"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const s3n = require("../lib");
// tslint:disable:object-literal-key-quotes
// tslint:disable:max-line-length
test('bucket without notifications', () => {
    const stack = new cdk.Stack();
    new s3.Bucket(stack, 'MyBucket');
    expect(stack).toMatchTemplate({
        'Resources': {
            'MyBucketF68F3FF0': {
                'Type': 'AWS::S3::Bucket',
                'DeletionPolicy': 'Retain',
                'UpdateReplacePolicy': 'Retain',
            },
        },
    });
});
test('when notification are added, a custom resource is provisioned + a lambda handler for it', () => {
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'MyBucket');
    const topic = new sns.Topic(stack, 'MyTopic');
    bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.SnsDestination(topic));
    expect(stack).toHaveResource('AWS::S3::Bucket');
    expect(stack).toHaveResource('AWS::Lambda::Function', { Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)' });
    expect(stack).toHaveResource('Custom::S3BucketNotifications');
});
test('when notification are added, you can tag the lambda', () => {
    const stack = new cdk.Stack();
    stack.node.applyAspect(new cdk.Tag('Lambda', 'AreTagged'));
    const bucket = new s3.Bucket(stack, 'MyBucket');
    const topic = new sns.Topic(stack, 'MyTopic');
    bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.SnsDestination(topic));
    expect(stack).toHaveResource('AWS::S3::Bucket');
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Tags: [{ Key: 'Lambda', Value: 'AreTagged' }],
        Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)'
    });
    expect(stack).toHaveResource('Custom::S3BucketNotifications');
});
test('bucketNotificationTarget is not called during synthesis', () => {
    const stack = new cdk.Stack();
    // notice the order here - topic is defined before bucket
    // but this shouldn't impact the fact that the topic policy includes
    // the bucket information
    const topic = new sns.Topic(stack, 'Topic');
    const bucket = new s3.Bucket(stack, 'MyBucket');
    bucket.addObjectCreatedNotification(new s3n.SnsDestination(topic));
    expect(stack).toHaveResourceLike('AWS::SNS::TopicPolicy', {
        'Topics': [
            {
                'Ref': 'TopicBFC7AF6E',
            },
        ],
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': 'sns:Publish',
                    'Condition': {
                        'ArnLike': {
                            'aws:SourceArn': {
                                'Fn::GetAtt': [
                                    'MyBucketF68F3FF0',
                                    'Arn',
                                ],
                            },
                        },
                    },
                    'Effect': 'Allow',
                    'Principal': {
                        'Service': 's3.amazonaws.com',
                    },
                    'Resource': {
                        'Ref': 'TopicBFC7AF6E',
                    },
                },
            ],
            'Version': '2012-10-17',
        },
    });
});
test('subscription types', () => {
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'TestBucket');
    const queueTarget = {
        bind: _ => ({
            type: s3.BucketNotificationDestinationType.QUEUE,
            arn: 'arn:aws:sqs:...',
        }),
    };
    const lambdaTarget = {
        bind: _ => ({
            type: s3.BucketNotificationDestinationType.LAMBDA,
            arn: 'arn:aws:lambda:...',
        }),
    };
    const topicTarget = {
        bind: _ => ({
            type: s3.BucketNotificationDestinationType.TOPIC,
            arn: 'arn:aws:sns:...',
        }),
    };
    bucket.addEventNotification(s3.EventType.OBJECT_CREATED, queueTarget);
    bucket.addEventNotification(s3.EventType.OBJECT_CREATED, lambdaTarget);
    bucket.addObjectRemovedNotification(topicTarget, { prefix: 'prefix' });
    expect(stack).toHaveResource('Custom::S3BucketNotifications', {
        'ServiceToken': {
            'Fn::GetAtt': [
                'BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691',
                'Arn',
            ],
        },
        'BucketName': {
            'Ref': 'TestBucket560B80BC',
        },
        'NotificationConfiguration': {
            'LambdaFunctionConfigurations': [
                {
                    'Events': [
                        's3:ObjectCreated:*',
                    ],
                    'LambdaFunctionArn': 'arn:aws:lambda:...',
                },
            ],
            'QueueConfigurations': [
                {
                    'Events': [
                        's3:ObjectCreated:*',
                    ],
                    'QueueArn': 'arn:aws:sqs:...',
                },
            ],
            'TopicConfigurations': [
                {
                    'Events': [
                        's3:ObjectRemoved:*',
                    ],
                    'TopicArn': 'arn:aws:sns:...',
                    'Filter': {
                        'Key': {
                            'FilterRules': [
                                {
                                    'Name': 'prefix',
                                    'Value': 'prefix',
                                },
                            ],
                        },
                    },
                },
            ],
        },
    });
});
test('multiple subscriptions of the same type', () => {
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'TestBucket');
    bucket.addEventNotification(s3.EventType.OBJECT_REMOVED_DELETE, {
        bind: _ => ({
            type: s3.BucketNotificationDestinationType.QUEUE,
            arn: 'arn:aws:sqs:...:queue1',
        }),
    });
    bucket.addEventNotification(s3.EventType.OBJECT_REMOVED_DELETE, {
        bind: _ => ({
            type: s3.BucketNotificationDestinationType.QUEUE,
            arn: 'arn:aws:sqs:...:queue2',
        }),
    });
    expect(stack).toHaveResource('Custom::S3BucketNotifications', {
        'ServiceToken': {
            'Fn::GetAtt': [
                'BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691',
                'Arn',
            ],
        },
        'BucketName': {
            'Ref': 'TestBucket560B80BC',
        },
        'NotificationConfiguration': {
            'QueueConfigurations': [
                {
                    'Events': [
                        's3:ObjectRemoved:Delete',
                    ],
                    'QueueArn': 'arn:aws:sqs:...:queue1',
                },
                {
                    'Events': [
                        's3:ObjectRemoved:Delete',
                    ],
                    'QueueArn': 'arn:aws:sqs:...:queue2',
                },
            ],
        },
    });
});
test('prefix/suffix filters', () => {
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'TestBucket');
    const bucketNotificationTarget = {
        type: s3.BucketNotificationDestinationType.QUEUE,
        arn: 'arn:aws:sqs:...',
    };
    bucket.addEventNotification(s3.EventType.OBJECT_REMOVED_DELETE, { bind: _ => bucketNotificationTarget }, { prefix: 'images/', suffix: '.jpg' });
    expect(stack).toHaveResource('Custom::S3BucketNotifications', {
        'ServiceToken': {
            'Fn::GetAtt': [
                'BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691',
                'Arn',
            ],
        },
        'BucketName': {
            'Ref': 'TestBucket560B80BC',
        },
        'NotificationConfiguration': {
            'QueueConfigurations': [
                {
                    'Events': [
                        's3:ObjectRemoved:Delete',
                    ],
                    'Filter': {
                        'Key': {
                            'FilterRules': [
                                {
                                    'Name': 'suffix',
                                    'Value': '.jpg',
                                },
                                {
                                    'Name': 'prefix',
                                    'Value': 'images/',
                                },
                            ],
                        },
                    },
                    'QueueArn': 'arn:aws:sqs:...',
                },
            ],
        },
    });
});
test('a notification destination can specify a set of dependencies that must be resolved before the notifications resource is created', () => {
    const stack = new cdk.Stack();
    const bucket = new s3.Bucket(stack, 'Bucket');
    const dependent = new cdk.CfnResource(stack, 'Dependent', { type: 'DependOnMe' });
    const dest = {
        bind: () => ({
            arn: 'arn',
            type: s3.BucketNotificationDestinationType.QUEUE,
            dependencies: [dependent],
        }),
    };
    bucket.addObjectCreatedNotification(dest);
    cdk.ConstructNode.prepare(stack.node);
    expect(assert_1.SynthUtils.synthesize(stack).template.Resources.BucketNotifications8F2E257D).toEqual({
        Type: 'Custom::S3BucketNotifications',
        Properties: {
            ServiceToken: { 'Fn::GetAtt': ['BucketNotificationsHandler050a0587b7544547bf325f094a3db8347ECC3691', 'Arn'] },
            BucketName: { Ref: 'Bucket83908E77' },
            NotificationConfiguration: { QueueConfigurations: [{ Events: ['s3:ObjectCreated:*'], QueueArn: 'arn' }] },
        },
        DependsOn: ['Dependent'],
    });
});
describe('CloudWatch Events', () => {
    test('onCloudTrailPutObject contains the Bucket ARN itself when path is undefined', () => {
        const stack = new cdk.Stack();
        const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
            bucketName: 'MyBucket',
        });
        bucket.onCloudTrailPutObject('PutRule', {
            target: {
                bind: () => ({ arn: 'ARN', id: '' }),
            },
        });
        expect(stack).toHaveResourceLike('AWS::Events::Rule', {
            'EventPattern': {
                'source': [
                    'aws.s3',
                ],
                'detail': {
                    'eventName': [
                        'PutObject',
                    ],
                    'resources': {
                        'ARN': [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            'Ref': 'AWS::Partition',
                                        },
                                        ':s3:::MyBucket',
                                    ],
                                ],
                            },
                        ],
                    },
                },
            },
            'State': 'ENABLED',
        });
    });
    test("onCloudTrailPutObject contains the path when it's provided", () => {
        const stack = new cdk.Stack();
        const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
            bucketName: 'MyBucket',
        });
        bucket.onCloudTrailPutObject('PutRule', {
            target: {
                bind: () => ({ arn: 'ARN', id: '' }),
            },
            paths: ['my/path.zip'],
        });
        expect(stack).toHaveResourceLike('AWS::Events::Rule', {
            'EventPattern': {
                'source': [
                    'aws.s3',
                ],
                'detail': {
                    'eventName': [
                        'PutObject',
                    ],
                    'resources': {
                        'ARN': [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            'Ref': 'AWS::Partition',
                                        },
                                        ':s3:::MyBucket/my/path.zip',
                                    ],
                                ],
                            },
                        ],
                    },
                },
            },
            'State': 'ENABLED',
        });
    });
    test('onCloudTrailWriteObject matches on events CompleteMultipartUpload, CopyObject, and PutObject', () => {
        const stack = new cdk.Stack();
        const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
            bucketName: 'MyBucket',
        });
        bucket.onCloudTrailWriteObject('OnCloudTrailWriteObjectRule', {
            target: {
                bind: () => ({ arn: 'ARN', id: '' }),
            },
        });
        expect(stack).toHaveResourceLike('AWS::Events::Rule', {
            'EventPattern': {
                'source': [
                    'aws.s3',
                ],
                'detail': {
                    'eventName': [
                        'CompleteMultipartUpload',
                        'CopyObject',
                        'PutObject',
                    ],
                },
            },
            'State': 'ENABLED',
        });
    });
    test('onCloudTrailWriteObject matches on the requestParameter bucketName when the path is not provided', () => {
        const stack = new cdk.Stack();
        const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
            bucketName: 'MyBucket',
        });
        bucket.onCloudTrailWriteObject('OnCloudTrailWriteObjectRule', {
            target: {
                bind: () => ({ arn: 'ARN', id: '' }),
            },
        });
        expect(stack).toHaveResourceLike('AWS::Events::Rule', {
            'EventPattern': {
                'source': [
                    'aws.s3',
                ],
                'detail': {
                    'requestParameters': {
                        'bucketName': [
                            bucket.bucketName,
                        ],
                    },
                },
            },
        });
    });
    test('onCloudTrailWriteObject matches on the requestParameters bucketName and key when the path is provided', () => {
        const stack = new cdk.Stack();
        const bucket = s3.Bucket.fromBucketAttributes(stack, 'Bucket', {
            bucketName: 'MyBucket',
        });
        bucket.onCloudTrailWriteObject('OnCloudTrailWriteObjectRule', {
            target: {
                bind: () => ({ arn: 'ARN', id: '' }),
            },
            paths: ['my/path.zip'],
        });
        expect(stack).toHaveResourceLike('AWS::Events::Rule', {
            'EventPattern': {
                'source': [
                    'aws.s3',
                ],
                'detail': {
                    'requestParameters': {
                        'bucketName': [
                            bucket.bucketName,
                        ],
                        'key': [
                            'my/path.zip',
                        ],
                    },
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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