"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
/**
 * Use an SQS queue as a bucket notification destination
 */
class SqsDestination {
    constructor(queue) {
        this.queue = queue;
    }
    /**
     * Allows using SQS queues as destinations for bucket notifications.
     * Use `bucket.onEvent(event, queue)` to subscribe.
     */
    bind(_scope, bucket) {
        this.queue.grantSendMessages(new iam.ServicePrincipal('s3.amazonaws.com', {
            conditions: {
                ArnLike: { 'aws:SourceArn': bucket.bucketArn },
            },
        }));
        // if this queue is encrypted, we need to allow S3 to read messages since that's how
        // it verifies that the notification destination configuration is valid.
        if (this.queue.encryptionMasterKey) {
            this.queue.encryptionMasterKey.addToResourcePolicy(new iam.PolicyStatement({
                principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                actions: ['kms:GenerateDataKey*', 'kms:Decrypt'],
                resources: ['*'],
            }), /* allowNoOp */ false);
        }
        return {
            arn: this.queue.queueArn,
            type: s3.BucketNotificationDestinationType.QUEUE,
            dependencies: [this.queue],
        };
    }
}
exports.SqsDestination = SqsDestination;
//# sourceMappingURL=data:application/json;base64,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