import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as notifications from '@aws-cdk/aws-codestarnotifications';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import { DockerImageAssetProps } from '@aws-cdk/aws-ecr-assets';
import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as s3 from '@aws-cdk/aws-s3';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { Duration, IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IArtifacts } from './artifacts';
import { BuildSpec } from './build-spec';
import { Cache } from './cache';
import { CfnProject } from './codebuild.generated';
import { IFileSystemLocation } from './file-location';
import { LoggingOptions } from './project-logs';
import { ISource } from './source';
import { Construct as CoreConstruct } from '@aws-cdk/core';
/**
 * The type returned from {@link IProject#enableBatchBuilds}.
 *
 * @stability stable
 */
export interface BatchBuildConfig {
    /**
     * The IAM batch service Role of this Project.
     *
     * @stability stable
     */
    readonly role: iam.IRole;
}
/**
 * Location of a PEM certificate on S3.
 *
 * @stability stable
 */
export interface BuildEnvironmentCertificate {
    /**
     * The bucket where the certificate is.
     *
     * @stability stable
     */
    readonly bucket: s3.IBucket;
    /**
     * The full path and name of the key file.
     *
     * @stability stable
     */
    readonly objectKey: string;
}
/**
 * Additional options to pass to the notification rule.
 *
 * @stability stable
 */
export interface ProjectNotifyOnOptions extends notifications.NotificationRuleOptions {
    /**
     * A list of event types associated with this notification rule for CodeBuild Project.
     *
     * For a complete list of event types and IDs, see Notification concepts in the Developer Tools Console User Guide.
     *
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#concepts-api
     * @stability stable
     */
    readonly events: ProjectNotificationEvents[];
}
/**
 * @stability stable
 */
export interface IProject extends IResource, iam.IGrantable, ec2.IConnectable, notifications.INotificationRuleSource {
    /**
     * The ARN of this Project.
     *
     * @stability stable
     * @attribute true
     */
    readonly projectArn: string;
    /**
     * The human-visible name of this Project.
     *
     * @stability stable
     * @attribute true
     */
    readonly projectName: string;
    /**
     * The IAM service Role of this Project.
     *
     * Undefined for imported Projects.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * Enable batch builds.
     *
     * Returns an object contining the batch service role if batch builds
     * could be enabled.
     *
     * @stability stable
     */
    enableBatchBuilds(): BatchBuildConfig | undefined;
    /**
     * @stability stable
     */
    addToRolePolicy(policyStatement: iam.PolicyStatement): void;
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when the build project state changes.
     *
     * You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *     const rule = project.onStateChange('OnBuildStarted', { target });
     *     rule.addEventPattern({
     *       detail: {
     *         'build-status': [
     *           "IN_PROGRESS",
     *           "SUCCEEDED",
     *           "FAILED",
     *           "STOPPED"
     *         ]
     *       }
     *     });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onPhaseChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * @stability stable
     */
    onBuildStarted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * @stability stable
     */
    onBuildFailed(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * @stability stable
     */
    onBuildSucceeded(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * @param metricName The name of the metric.
     * @param props Customization properties.
     * @returns a CloudWatch metric associated with this build project.
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     * @stability stable
     */
    metricDuration(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricSucceededBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds that failed because of client error or because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricFailedBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Defines a CodeStar Notification rule triggered when the project events emitted by you specified, it very similar to `onEvent` API.
     *
     * You can also use the methods `notifyOnBuildSucceeded` and
     * `notifyOnBuildFailed` to define rules for these specific event emitted.
     *
     * @param id The logical identifier of the CodeStar Notifications rule that will be created.
     * @param target The target to register for the CodeStar Notifications destination.
     * @param options Customization options for CodeStar Notifications rule.
     * @returns CodeStar Notifications rule associated with this build project.
     * @stability stable
     */
    notifyOn(id: string, target: notifications.INotificationRuleTarget, options: ProjectNotifyOnOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar notification rule which triggers when a build completes successfully.
     *
     * @stability stable
     */
    notifyOnBuildSucceeded(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar notification rule which triggers when a build fails.
     *
     * @stability stable
     */
    notifyOnBuildFailed(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
}
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
declare abstract class ProjectBase extends Resource implements IProject {
    abstract readonly grantPrincipal: iam.IPrincipal;
    /** The ARN of this Project. */
    abstract readonly projectArn: string;
    /** The human-visible name of this Project. */
    abstract readonly projectName: string;
    /** The IAM service Role of this Project. */
    abstract readonly role?: iam.IRole;
    /**
     * Actual connections object for this Project.
     * May be unset, in which case this Project is not configured to use a VPC.
     * @internal
     */
    protected _connections: ec2.Connections | undefined;
    /**
     * Access the Connections object.
     *
     * Will fail if this Project does not have a VPC set.
     *
     * @stability stable
     */
    get connections(): ec2.Connections;
    enableBatchBuilds(): BatchBuildConfig | undefined;
    /**
     * Add a permission only if there's a policy attached.
     *
     * @param statement The permissions statement to add.
     * @stability stable
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule triggered when the build project state changes.
     *
     * You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *     const rule = project.onStateChange('OnBuildStarted', { target });
     *     rule.addEventPattern({
     *       detail: {
     *         'build-status': [
     *           "IN_PROGRESS",
     *           "SUCCEEDED",
     *           "FAILED",
     *           "STOPPED"
     *         ]
     *       }
     *     });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onStateChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onPhaseChange(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @stability stable
     */
    onBuildStarted(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @stability stable
     */
    onBuildFailed(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @stability stable
     */
    onBuildSucceeded(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * @param metricName The name of the metric.
     * @param props Customization properties.
     * @returns a CloudWatch metric associated with this build project.
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     * @stability stable
     */
    metricDuration(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricSucceededBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Measures the number of builds that failed because of client error or because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricFailedBuilds(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Defines a CodeStar Notification rule triggered when the project events emitted by you specified, it very similar to `onEvent` API.
     *
     * You can also use the methods `notifyOnBuildSucceeded` and
     * `notifyOnBuildFailed` to define rules for these specific event emitted.
     *
     * @stability stable
     */
    notifyOn(id: string, target: notifications.INotificationRuleTarget, options: ProjectNotifyOnOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar notification rule which triggers when a build completes successfully.
     *
     * @stability stable
     */
    notifyOnBuildSucceeded(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Defines a CodeStar notification rule which triggers when a build fails.
     *
     * @stability stable
     */
    notifyOnBuildFailed(id: string, target: notifications.INotificationRuleTarget, options?: notifications.NotificationRuleOptions): notifications.INotificationRule;
    /**
     * Returns a source configuration for notification rule.
     *
     * @stability stable
     */
    bindAsNotificationRuleSource(_scope: Construct): notifications.NotificationRuleSourceConfig;
    private cannedMetric;
}
/**
 * @stability stable
 */
export interface CommonProjectProps {
    /**
     * A description of the project.
     *
     * Use the description to identify the purpose
     * of the project.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
    /**
     * Filename or contents of buildspec in JSON format.
     *
     * @default - Empty buildspec.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-example
     * @stability stable
     */
    readonly buildSpec?: BuildSpec;
    /**
     * Service Role to assume while running the build.
     *
     * @default - A role will be created.
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * Encryption key to use to read and write artifacts.
     *
     * @default - The AWS-managed CMK for Amazon Simple Storage Service (Amazon S3) is used.
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Caching strategy to use.
     *
     * @default Cache.none
     * @stability stable
     */
    readonly cache?: Cache;
    /**
     * Build environment to use for the build.
     *
     * @default BuildEnvironment.LinuxBuildImage.STANDARD_1_0
     * @stability stable
     */
    readonly environment?: BuildEnvironment;
    /**
     * Indicates whether AWS CodeBuild generates a publicly accessible URL for your project's build badge.
     *
     * For more information, see Build Badges Sample
     * in the AWS CodeBuild User Guide.
     *
     * @default false
     * @stability stable
     */
    readonly badge?: boolean;
    /**
     * The number of minutes after which AWS CodeBuild stops the build if it's not complete.
     *
     * For valid values, see the timeoutInMinutes field in the AWS
     * CodeBuild User Guide.
     *
     * @default Duration.hours(1)
     * @stability stable
     */
    readonly timeout?: Duration;
    /**
     * Additional environment variables to add to the build environment.
     *
     * @default - No additional environment variables are specified.
     * @stability stable
     */
    readonly environmentVariables?: {
        [name: string]: BuildEnvironmentVariable;
    };
    /**
     * Whether to check for the presence of any secrets in the environment variables of the default type, BuildEnvironmentVariableType.PLAINTEXT. Since using a secret for the value of that kind of variable would result in it being displayed in plain text in the AWS Console, the construct will throw an exception if it detects a secret was passed there. Pass this property as false if you want to skip this validation, and keep using a secret in a plain text environment variable.
     *
     * @default true
     * @stability stable
     */
    readonly checkSecretsInPlainTextEnvVariables?: boolean;
    /**
     * The physical, human-readable name of the CodeBuild Project.
     *
     * @default - Name is automatically generated.
     * @stability stable
     */
    readonly projectName?: string;
    /**
     * VPC network to place codebuild network interfaces.
     *
     * Specify this if the codebuild project needs to access resources in a VPC.
     *
     * @default - No VPC is specified.
     * @stability stable
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - All private subnets.
     * @stability stable
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * What security group to associate with the codebuild project's network interfaces.
     *
     * If no security group is identified, one will be created automatically.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default - Security group will be automatically created.
     * @stability stable
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Whether to allow the CodeBuild to send all network traffic.
     *
     * If set to false, you must individually add traffic rules to allow the
     * CodeBuild project to connect to network targets.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default true
     * @stability stable
     */
    readonly allowAllOutbound?: boolean;
    /**
     * An  ProjectFileSystemLocation objects for a CodeBuild build project.
     *
     * A ProjectFileSystemLocation object specifies the identifier, location, mountOptions, mountPoint,
     * and type of a file system created using Amazon Elastic File System.
     *
     * @default - no file system locations
     * @stability stable
     */
    readonly fileSystemLocations?: IFileSystemLocation[];
    /**
     * Add permissions to this project's role to create and use test report groups with name starting with the name of this project.
     *
     * That is the standard report group that gets created when a simple name
     * (in contrast to an ARN)
     * is used in the 'reports' section of the buildspec of this project.
     * This is usually harmless, but you can turn these off if you don't plan on using test
     * reports in this project.
     *
     * @default true
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/test-report-group-naming.html
     * @stability stable
     */
    readonly grantReportGroupPermissions?: boolean;
    /**
     * Information about logs for the build project.
     *
     * A project can create logs in Amazon CloudWatch Logs, an S3 bucket, or both.
     *
     * @default - no log configuration is set
     * @stability stable
     */
    readonly logging?: LoggingOptions;
    /**
     * The number of minutes after which AWS CodeBuild stops the build if it's still in queue.
     *
     * For valid values, see the timeoutInMinutes field in the AWS
     * CodeBuild User Guide.
     *
     * @default - no queue timeout is set
     * @stability stable
     */
    readonly queuedTimeout?: Duration;
    /**
     * Maximum number of concurrent builds.
     *
     * Minimum value is 1 and maximum is account build limit.
     *
     * @default - no explicit limit is set
     * @stability stable
     */
    readonly concurrentBuildLimit?: number;
}
/**
 * @stability stable
 */
export interface ProjectProps extends CommonProjectProps {
    /**
     * The source of the build.
     *
     * *Note*: if {@link NoSource} is given as the source,
     * then you need to provide an explicit `buildSpec`.
     *
     * @default - NoSource
     * @stability stable
     */
    readonly source?: ISource;
    /**
     * Defines where build artifacts will be stored.
     *
     * Could be: PipelineBuildArtifacts, NoArtifacts and S3Artifacts.
     *
     * @default NoArtifacts
     * @stability stable
     */
    readonly artifacts?: IArtifacts;
    /**
     * The secondary sources for the Project.
     *
     * Can be also added after the Project has been created by using the {@link Project#addSecondarySource} method.
     *
     * @default - No secondary sources.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @stability stable
     */
    readonly secondarySources?: ISource[];
    /**
     * The secondary artifacts for the Project.
     *
     * Can also be added after the Project has been created by using the {@link Project#addSecondaryArtifact} method.
     *
     * @default - No secondary artifacts.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @stability stable
     */
    readonly secondaryArtifacts?: IArtifacts[];
}
/**
 * The extra options passed to the {@link IProject.bindToCodePipeline} method.
 *
 * @stability stable
 */
export interface BindToCodePipelineOptions {
    /**
     * The artifact bucket that will be used by the action that invokes this project.
     *
     * @stability stable
     */
    readonly artifactBucket: s3.IBucket;
}
/**
 * A representation of a CodeBuild Project.
 *
 * @stability stable
 */
export declare class Project extends ProjectBase {
    /**
     * @stability stable
     */
    static fromProjectArn(scope: Construct, id: string, projectArn: string): IProject;
    /**
     * Import a Project defined either outside the CDK, or in a different CDK Stack (and exported using the {@link export} method).
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param projectName the name of the project to import.
     * @returns a reference to the existing Project
     * @stability stable
     * @note if you're importing a CodeBuild Project for use
     * in a CodePipeline, make sure the existing Project
     * has permissions to access the S3 Bucket of that Pipeline -
     * otherwise, builds in that Pipeline will always fail.
     */
    static fromProjectName(scope: Construct, id: string, projectName: string): IProject;
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable}, which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty}, which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables.
     * @param validateNoPlainTextSecrets whether to throw an exception if any of the plain text environment variables contain secrets, defaults to 'false'.
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     * @stability stable
     */
    static serializeEnvVariables(environmentVariables: {
        [name: string]: BuildEnvironmentVariable;
    }, validateNoPlainTextSecrets?: boolean, principal?: iam.IGrantable): CfnProject.EnvironmentVariableProperty[];
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * The IAM role for this project.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
    /**
     * The ARN of the project.
     *
     * @stability stable
     */
    readonly projectArn: string;
    /**
     * The name of the project.
     *
     * @stability stable
     */
    readonly projectName: string;
    private readonly source;
    private readonly buildImage;
    private readonly _secondarySources;
    private readonly _secondarySourceVersions;
    private readonly _secondaryArtifacts;
    private _encryptionKey?;
    private readonly _fileSystemLocations;
    private _batchServiceRole?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ProjectProps);
    /**
     * Enable batch builds.
     *
     * Returns an object contining the batch service role if batch builds
     * could be enabled.
     *
     * @stability stable
     */
    enableBatchBuilds(): BatchBuildConfig | undefined;
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @stability stable
     */
    addSecondarySource(secondarySource: ISource): void;
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add.
     * @stability stable
     */
    addFileSystemLocation(fileSystemLocation: IFileSystemLocation): void;
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @stability stable
     */
    addSecondaryArtifact(secondaryArtifact: IArtifacts): void;
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in.
     * @param options additional options for the binding.
     * @stability stable
     */
    bindToCodePipeline(_scope: CoreConstruct, options: BindToCodePipelineOptions): void;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     * @override true
     */
    protected validate(): string[];
    private set encryptionKey(value);
    private createLoggingPermission;
    private renderEnvironment;
    private renderFileSystemLocations;
    private renderSecondarySources;
    private renderSecondarySourceVersions;
    private renderSecondaryArtifacts;
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    private configureVpc;
    private renderLoggingConfiguration;
    private addVpcRequiredPermissions;
    private validateCodePipelineSettings;
}
/**
 * Build machine compute type.
 *
 * @stability stable
 */
export declare enum ComputeType {
    /**
     * @stability stable
     */
    SMALL = "BUILD_GENERAL1_SMALL",
    /**
     * @stability stable
     */
    MEDIUM = "BUILD_GENERAL1_MEDIUM",
    /**
     * @stability stable
     */
    LARGE = "BUILD_GENERAL1_LARGE",
    /**
     * @stability stable
     */
    X2_LARGE = "BUILD_GENERAL1_2XLARGE"
}
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 *
 * @stability stable
 */
export declare enum ImagePullPrincipalType {
    /**
     * CODEBUILD specifies that CodeBuild uses its own identity when pulling the image.
     *
     * This means the resource policy of the ECR repository that hosts the image will be modified to trust
     * CodeBuild's service principal.
     * This is the required principal type when using CodeBuild's pre-defined images.
     *
     * @stability stable
     */
    CODEBUILD = "CODEBUILD",
    /**
     * SERVICE_ROLE specifies that AWS CodeBuild uses the project's role when pulling the image.
     *
     * The role will be granted pull permissions on the ECR repository hosting the image.
     *
     * @stability stable
     */
    SERVICE_ROLE = "SERVICE_ROLE"
}
/**
 * @stability stable
 */
export interface BuildEnvironment {
    /**
     * The image used for the builds.
     *
     * @default LinuxBuildImage.STANDARD_1_0
     * @stability stable
     */
    readonly buildImage?: IBuildImage;
    /**
     * The type of compute to use for this build.
     *
     * See the {@link ComputeType} enum for the possible values.
     *
     * @default taken from {@link #buildImage#defaultComputeType}
     * @stability stable
     */
    readonly computeType?: ComputeType;
    /**
     * Indicates how the project builds Docker images.
     *
     * Specify true to enable
     * running the Docker daemon inside a Docker container. This value must be
     * set to true only if this build project will be used to build Docker
     * images, and the specified build environment image is not one provided by
     * AWS CodeBuild with Docker support. Otherwise, all associated builds that
     * attempt to interact with the Docker daemon will fail.
     *
     * @default false
     * @stability stable
     */
    readonly privileged?: boolean;
    /**
     * The location of the PEM-encoded certificate for the build project.
     *
     * @default - No external certificate is added to the project
     * @stability stable
     */
    readonly certificate?: BuildEnvironmentCertificate;
    /**
     * The environment variables that your builds can use.
     *
     * @stability stable
     */
    readonly environmentVariables?: {
        [name: string]: BuildEnvironmentVariable;
    };
}
/**
 * Represents a Docker image used for the CodeBuild Project builds.
 *
 * Use the concrete subclasses, either:
 * {@link LinuxBuildImage} or {@link WindowsBuildImage}.
 *
 * @stability stable
 */
export interface IBuildImage {
    /**
     * The type of build environment.
     *
     * @stability stable
     */
    readonly type: string;
    /**
     * The Docker image identifier that the build environment uses.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @stability stable
     */
    readonly imageId: string;
    /**
     * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
     *
     * @stability stable
     */
    readonly defaultComputeType: ComputeType;
    /**
     * The type of principal that CodeBuild will use to pull this build Docker image.
     *
     * @default ImagePullPrincipalType.SERVICE_ROLE
     * @stability stable
     */
    readonly imagePullPrincipalType?: ImagePullPrincipalType;
    /**
     * The secretsManagerCredentials for access to a private registry.
     *
     * @default no credentials will be used
     * @stability stable
     */
    readonly secretsManagerCredentials?: secretsmanager.ISecret;
    /**
     * An optional ECR repository that the image is hosted in.
     *
     * @default no repository
     * @stability stable
     */
    readonly repository?: ecr.IRepository;
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @param buildEnvironment the current build environment.
     * @stability stable
     */
    validate(buildEnvironment: BuildEnvironment): string[];
    /**
     * Make a buildspec to run the indicated script.
     *
     * @stability stable
     */
    runScriptBuildspec(entrypoint: string): BuildSpec;
}
/**
 * Optional arguments to {@link IBuildImage.binder} - currently empty.
 *
 * @stability stable
 */
export interface BuildImageBindOptions {
}
/**
 * The return type from {@link IBuildImage.binder} - currently empty.
 *
 * @stability stable
 */
export interface BuildImageConfig {
}
/**
 * A variant of {@link IBuildImage} that allows binding to the project.
 *
 * @stability stable
 */
export interface IBindableBuildImage extends IBuildImage {
    /**
     * Function that allows the build image access to the construct tree.
     *
     * @stability stable
     */
    bind(scope: CoreConstruct, project: IProject, options: BuildImageBindOptions): BuildImageConfig;
}
/**
 * The options when creating a CodeBuild Docker build image using {@link LinuxBuildImage.fromDockerRegistry} or {@link WindowsBuildImage.fromDockerRegistry}.
 *
 * @stability stable
 */
export interface DockerImageOptions {
    /**
     * The credentials, stored in Secrets Manager, used for accessing the repository holding the image, if the repository is private.
     *
     * @default no credentials will be used (we assume the repository is public)
     * @stability stable
     */
    readonly secretsManagerCredentials?: secretsmanager.ISecret;
}
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @stability stable
 */
export declare class LinuxBuildImage implements IBuildImage {
    /**
     * @stability stable
     */
    static readonly STANDARD_1_0: IBuildImage;
    /**
     * @stability stable
     */
    static readonly STANDARD_2_0: IBuildImage;
    /**
     * @stability stable
     */
    static readonly STANDARD_3_0: IBuildImage;
    /**
     * The `aws/codebuild/standard:4.0` build image.
     *
     * @stability stable
     */
    static readonly STANDARD_4_0: IBuildImage;
    /**
     * The `aws/codebuild/standard:5.0` build image.
     *
     * @stability stable
     */
    static readonly STANDARD_5_0: IBuildImage;
    /**
     * @stability stable
     */
    static readonly AMAZON_LINUX_2: IBuildImage;
    /**
     * @stability stable
     */
    static readonly AMAZON_LINUX_2_2: IBuildImage;
    /**
     * The Amazon Linux 2 x86_64 standard image, version `3.0`.
     *
     * @stability stable
     */
    static readonly AMAZON_LINUX_2_3: IBuildImage;
    /**
     * @stability stable
     */
    static readonly AMAZON_LINUX_2_ARM: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_BASE: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_ANDROID_JAVA8_24_4_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_ANDROID_JAVA8_26_1_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_DOCKER_17_09_0: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_DOCKER_18_09_0: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_GOLANG_1_10: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_GOLANG_1_11: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_OPEN_JDK_8: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_OPEN_JDK_9: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_OPEN_JDK_11: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_NODEJS_10_14_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_NODEJS_10_1_0: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_NODEJS_8_11_0: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_NODEJS_6_3_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PHP_5_6: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PHP_7_0: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PHP_7_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PYTHON_3_7_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PYTHON_3_6_5: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PYTHON_3_5_2: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PYTHON_3_4_5: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PYTHON_3_3_6: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_PYTHON_2_7_12: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_RUBY_2_5_3: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_RUBY_2_5_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_RUBY_2_3_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_RUBY_2_2_5: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_DOTNET_CORE_1_1: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_DOTNET_CORE_2_0: IBuildImage;
    /**
     * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
     */
    static readonly UBUNTU_14_04_DOTNET_CORE_2_1: IBuildImage;
    /**
     * @returns a Linux build image from a Docker Hub image.
     * @stability stable
     */
    static fromDockerRegistry(name: string, options?: DockerImageOptions): IBuildImage;
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @stability stable
     */
    static fromEcrRepository(repository: ecr.IRepository, tag?: string): IBuildImage;
    /**
     * Uses an Docker image asset as a Linux build image.
     *
     * @stability stable
     */
    static fromAsset(scope: Construct, id: string, props: DockerImageAssetProps): IBuildImage;
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @param id The image identifier.
     * @returns A Docker image provided by CodeBuild.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @stability stable
     * @example
     *
     * 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id: string): IBuildImage;
    private static codeBuildImage;
    /**
     * The type of build environment.
     *
     * @stability stable
     */
    readonly type = "LINUX_CONTAINER";
    /**
     * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
     *
     * @stability stable
     */
    readonly defaultComputeType = ComputeType.SMALL;
    /**
     * The Docker image identifier that the build environment uses.
     *
     * @stability stable
     */
    readonly imageId: string;
    /**
     * The type of principal that CodeBuild will use to pull this build Docker image.
     *
     * @stability stable
     */
    readonly imagePullPrincipalType?: ImagePullPrincipalType;
    /**
     * The secretsManagerCredentials for access to a private registry.
     *
     * @stability stable
     */
    readonly secretsManagerCredentials?: secretsmanager.ISecret;
    /**
     * An optional ECR repository that the image is hosted in.
     *
     * @stability stable
     */
    readonly repository?: ecr.IRepository;
    private constructor();
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @stability stable
     */
    validate(_: BuildEnvironment): string[];
    /**
     * Make a buildspec to run the indicated script.
     *
     * @stability stable
     */
    runScriptBuildspec(entrypoint: string): BuildSpec;
}
/**
 * Environment type for Windows Docker images.
 *
 * @stability stable
 */
export declare enum WindowsImageType {
    /**
     * The standard environment type, WINDOWS_CONTAINER.
     *
     * @stability stable
     */
    STANDARD = "WINDOWS_CONTAINER",
    /**
     * The WINDOWS_SERVER_2019_CONTAINER environment type.
     *
     * @stability stable
     */
    SERVER_2019 = "WINDOWS_SERVER_2019_CONTAINER"
}
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @stability stable
 */
export declare class WindowsBuildImage implements IBuildImage {
    /**
     * (deprecated) Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
     *
     * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
     */
    static readonly WIN_SERVER_CORE_2016_BASE: IBuildImage;
    /**
     * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is based off Windows Server Core 2016.
     *
     * @stability stable
     */
    static readonly WINDOWS_BASE_2_0: IBuildImage;
    /**
     * The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is based off Windows Server Core 2019.
     *
     * @stability stable
     */
    static readonly WIN_SERVER_CORE_2019_BASE: IBuildImage;
    /**
     * @returns a Windows build image from a Docker Hub image.
     * @stability stable
     */
    static fromDockerRegistry(name: string, options?: DockerImageOptions, imageType?: WindowsImageType): IBuildImage;
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @stability stable
     */
    static fromEcrRepository(repository: ecr.IRepository, tag?: string, imageType?: WindowsImageType): IBuildImage;
    /**
     * Uses an Docker image asset as a Windows build image.
     *
     * @stability stable
     */
    static fromAsset(scope: Construct, id: string, props: DockerImageAssetProps, imageType?: WindowsImageType): IBuildImage;
    /**
     * The type of build environment.
     *
     * @stability stable
     */
    readonly type: string;
    /**
     * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
     *
     * @stability stable
     */
    readonly defaultComputeType = ComputeType.MEDIUM;
    /**
     * The Docker image identifier that the build environment uses.
     *
     * @stability stable
     */
    readonly imageId: string;
    /**
     * The type of principal that CodeBuild will use to pull this build Docker image.
     *
     * @stability stable
     */
    readonly imagePullPrincipalType?: ImagePullPrincipalType;
    /**
     * The secretsManagerCredentials for access to a private registry.
     *
     * @stability stable
     */
    readonly secretsManagerCredentials?: secretsmanager.ISecret;
    /**
     * An optional ECR repository that the image is hosted in.
     *
     * @stability stable
     */
    readonly repository?: ecr.IRepository;
    private constructor();
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @stability stable
     */
    validate(buildEnvironment: BuildEnvironment): string[];
    /**
     * Make a buildspec to run the indicated script.
     *
     * @stability stable
     */
    runScriptBuildspec(entrypoint: string): BuildSpec;
}
/**
 * @stability stable
 */
export interface BuildEnvironmentVariable {
    /**
     * The type of environment variable.
     *
     * @default PlainText
     * @stability stable
     */
    readonly type?: BuildEnvironmentVariableType;
    /**
     * The value of the environment variable.
     *
     * For plain-text variables (the default), this is the literal value of variable.
     * For SSM parameter variables, pass the name of the parameter here (`parameterName` property of `IParameter`).
     * For SecretsManager variables secrets, pass either the secret name (`secretName` property of `ISecret`)
     * or the secret ARN (`secretArn` property of `ISecret`) here,
     * along with optional SecretsManager qualifiers separated by ':', like the JSON key, or the version or stage
     * (see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager for details).
     *
     * @stability stable
     */
    readonly value: any;
}
/**
 * @stability stable
 */
export declare enum BuildEnvironmentVariableType {
    /**
     * An environment variable in plaintext format.
     *
     * @stability stable
     */
    PLAINTEXT = "PLAINTEXT",
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     *
     * @stability stable
     */
    PARAMETER_STORE = "PARAMETER_STORE",
    /**
     * An environment variable stored in AWS Secrets Manager.
     *
     * @stability stable
     */
    SECRETS_MANAGER = "SECRETS_MANAGER"
}
/**
 * The list of event types for AWS Codebuild.
 *
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-buildproject
 * @stability stable
 */
export declare enum ProjectNotificationEvents {
    /**
     * Trigger notification when project build state failed.
     *
     * @stability stable
     */
    BUILD_FAILED = "codebuild-project-build-state-failed",
    /**
     * Trigger notification when project build state succeeded.
     *
     * @stability stable
     */
    BUILD_SUCCEEDED = "codebuild-project-build-state-succeeded",
    /**
     * Trigger notification when project build state in progress.
     *
     * @stability stable
     */
    BUILD_IN_PROGRESS = "codebuild-project-build-state-in-progress",
    /**
     * Trigger notification when project build state stopped.
     *
     * @stability stable
     */
    BUILD_STOPPED = "codebuild-project-build-state-stopped",
    /**
     * Trigger notification when project build phase failure.
     *
     * @stability stable
     */
    BUILD_PHASE_FAILED = "codebuild-project-build-phase-failure",
    /**
     * Trigger notification when project build phase success.
     *
     * @stability stable
     */
    BUILD_PHASE_SUCCEEDED = "codebuild-project-build-phase-success"
}
export {};
