"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiSpecProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const fs_extra_1 = require("fs-extra");
const lodash_1 = require("lodash");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
const parsed_spec_1 = require("./components/parsed-spec");
// initialize logger
const logger = (0, log4js_1.getLogger)();
/**
 * Project containing the OpenAPI spec, and a parsed spec for use by the CDK construct
 */
class OpenApiSpecProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.specChanged = true;
        // Store whether we've synthesized the project
        this.synthed = false;
        logger.trace("OpenApiSpecProject constructor");
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        this.specFileName = options.specFileName ?? "spec.yaml";
        this.parsedSpecFileName = options.parsedSpecFileName ?? ".parsed-spec.json";
        if (!this.parsedSpecFileName.endsWith(".json")) {
            throw new Error("Parsed spec file must end with .json");
        }
        this.specPath =
            options.overrideSpecPath ?? path.join(this.outdir, this.specFileName);
        this.parsedSpecPath = path.join(this.outdir, ...(options.parsedSpecOutdir ? [options.parsedSpecOutdir] : []), this.parsedSpecFileName);
        logger.debug(`specPath = "${this.specPath}"`);
        logger.debug(`parsedSpecPath = "${this.parsedSpecPath}"`);
        // Create a sample OpenAPI spec yaml if not defined
        if (!options.overrideSpecPath) {
            new projen_1.SampleFile(this, this.specFileName, {
                sourcePath: path.join(__dirname, "..", "..", "..", "samples", "spec.yaml"),
            });
        }
        // Check if there is already a .parsed-spec.json present
        const existingParsedSpecJson = (0, util_1.tryReadFileSync)(this.parsedSpecPath);
        if (existingParsedSpecJson != null) {
            // generate a new temporary parsed-spec
            const tmpParseSpecPath = `${this.parsedSpecPath}.tmp`;
            logger.trace(`Generating temp spec at "${tmpParseSpecPath}"`);
            parsed_spec_1.ParsedSpec.parse(this.specPath, tmpParseSpecPath);
            // load it
            const newParsedSpecJson = (0, util_1.tryReadFileSync)(tmpParseSpecPath);
            if (newParsedSpecJson != null) {
                const prevParseSpec = JSON.parse(existingParsedSpecJson);
                const newParsedSpec = JSON.parse(newParsedSpecJson);
                // check if spec changed and keep it
                logger.trace("Comparing previous and newly generated specs.");
                this.specChanged = !(0, lodash_1.isEqual)(prevParseSpec, newParsedSpec);
                logger.debug(`Spec Changed :: ${this.specChanged}`);
                // remove tmp parsed-spec file
                logger.trace("Removing temp spec file.");
                (0, fs_extra_1.unlinkSync)(tmpParseSpecPath);
            }
        }
        else {
            logger.debug(`No parsedSpec file found at "${this.parsedSpecPath}".`);
        }
        // Parse the spec to produce a consolidated, bundled spec which can be read by cdk constructs or other tooling
        new parsed_spec_1.ParsedSpec(this, {
            specPath: this.specPath,
            outputPath: this.parsedSpecPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            logger.trace("OpenApiSpecProject already synthed. Skipping...");
            return;
        }
        logger.trace("OpenApiSpecProject synth");
        super.synth();
        this.synthed = true;
    }
}
exports.OpenApiSpecProject = OpenApiSpecProject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3Blbi1hcGktc3BlYy1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3Byb2plY3Qvc3BlYy9vcGVuLWFwaS1zcGVjLXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUE7c0NBQ3NDO0FBQ3RDLDZCQUE2QjtBQUM3Qix1Q0FBc0M7QUFDdEMsbUNBQWlDO0FBQ2pDLG1DQUFtQztBQUNuQyxtQ0FBNkQ7QUFDN0QsMENBQWtEO0FBQ2xELDBEQUFzRDtBQUV0RCxvQkFBb0I7QUFDcEIsTUFBTSxNQUFNLEdBQUcsSUFBQSxrQkFBUyxHQUFFLENBQUM7QUEyQjNCOztHQUVHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSxnQkFBTztJQVk3QyxZQUFZLE9BQWtDO1FBQzVDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQU5ELGdCQUFXLEdBQVksSUFBSSxDQUFDO1FBRTVDLDhDQUE4QztRQUN0QyxZQUFPLEdBQVksS0FBSyxDQUFDO1FBSS9CLE1BQU0sQ0FBQyxLQUFLLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMvQywrR0FBK0c7UUFDL0csMENBQTBDO1FBQzFDLGFBQWE7UUFDYixJQUFJLENBQUMsV0FBVyxHQUFHLEVBQUUsQ0FBQztRQUV0QixJQUFJLENBQUMsWUFBWSxHQUFHLE9BQU8sQ0FBQyxZQUFZLElBQUksV0FBVyxDQUFDO1FBQ3hELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxPQUFPLENBQUMsa0JBQWtCLElBQUksbUJBQW1CLENBQUM7UUFFNUUsSUFBSSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDOUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO1NBQ3pEO1FBRUQsSUFBSSxDQUFDLFFBQVE7WUFDWCxPQUFPLENBQUMsZ0JBQWdCLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUN4RSxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQzdCLElBQUksQ0FBQyxNQUFNLEVBQ1gsR0FBRyxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQy9ELElBQUksQ0FBQyxrQkFBa0IsQ0FDeEIsQ0FBQztRQUVGLE1BQU0sQ0FBQyxLQUFLLENBQUMsZUFBZSxJQUFJLENBQUMsUUFBUSxHQUFHLENBQUMsQ0FBQztRQUM5QyxNQUFNLENBQUMsS0FBSyxDQUFDLHFCQUFxQixJQUFJLENBQUMsY0FBYyxHQUFHLENBQUMsQ0FBQztRQUUxRCxtREFBbUQ7UUFDbkQsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsRUFBRTtZQUM3QixJQUFJLG1CQUFVLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZLEVBQUU7Z0JBQ3RDLFVBQVUsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUNuQixTQUFTLEVBQ1QsSUFBSSxFQUNKLElBQUksRUFDSixJQUFJLEVBQ0osU0FBUyxFQUNULFdBQVcsQ0FDWjthQUNGLENBQUMsQ0FBQztTQUNKO1FBRUQsd0RBQXdEO1FBQ3hELE1BQU0sc0JBQXNCLEdBQUcsSUFBQSxzQkFBZSxFQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUNwRSxJQUFJLHNCQUFzQixJQUFJLElBQUksRUFBRTtZQUNsQyx1Q0FBdUM7WUFDdkMsTUFBTSxnQkFBZ0IsR0FBRyxHQUFHLElBQUksQ0FBQyxjQUFjLE1BQU0sQ0FBQztZQUN0RCxNQUFNLENBQUMsS0FBSyxDQUFDLDRCQUE0QixnQkFBZ0IsR0FBRyxDQUFDLENBQUM7WUFDOUQsd0JBQVUsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1lBRWxELFVBQVU7WUFDVixNQUFNLGlCQUFpQixHQUFHLElBQUEsc0JBQWUsRUFBQyxnQkFBZ0IsQ0FBQyxDQUFDO1lBRTVELElBQUksaUJBQWlCLElBQUksSUFBSSxFQUFFO2dCQUM3QixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLENBQUM7Z0JBQ3pELE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQztnQkFFcEQsb0NBQW9DO2dCQUNwQyxNQUFNLENBQUMsS0FBSyxDQUFDLCtDQUErQyxDQUFDLENBQUM7Z0JBQzlELElBQUksQ0FBQyxXQUFXLEdBQUcsQ0FBQyxJQUFBLGdCQUFPLEVBQUMsYUFBYSxFQUFFLGFBQWEsQ0FBQyxDQUFDO2dCQUMxRCxNQUFNLENBQUMsS0FBSyxDQUFDLG1CQUFtQixJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztnQkFFcEQsOEJBQThCO2dCQUM5QixNQUFNLENBQUMsS0FBSyxDQUFDLDBCQUEwQixDQUFDLENBQUM7Z0JBQ3pDLElBQUEscUJBQVUsRUFBQyxnQkFBZ0IsQ0FBQyxDQUFDO2FBQzlCO1NBQ0Y7YUFBTTtZQUNMLE1BQU0sQ0FBQyxLQUFLLENBQUMsZ0NBQWdDLElBQUksQ0FBQyxjQUFjLElBQUksQ0FBQyxDQUFDO1NBQ3ZFO1FBRUQsOEdBQThHO1FBQzlHLElBQUksd0JBQVUsQ0FBQyxJQUFJLEVBQUU7WUFDbkIsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ3ZCLFVBQVUsRUFBRSxJQUFJLENBQUMsY0FBYztTQUNoQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxLQUFLO1FBQ0gsbUhBQW1IO1FBQ25ILGdIQUFnSDtRQUNoSCwwREFBMEQ7UUFDMUQsSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQ2hCLE1BQU0sQ0FBQyxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQztZQUNoRSxPQUFPO1NBQ1I7UUFDRCxNQUFNLENBQUMsS0FBSyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDekMsS0FBSyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWQsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUM7SUFDdEIsQ0FBQztDQUNGO0FBdkdELGdEQXVHQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgdW5saW5rU3luYyB9IGZyb20gXCJmcy1leHRyYVwiO1xuaW1wb3J0IHsgaXNFcXVhbCB9IGZyb20gXCJsb2Rhc2hcIjtcbmltcG9ydCB7IGdldExvZ2dlciB9IGZyb20gXCJsb2c0anNcIjtcbmltcG9ydCB7IFByb2plY3QsIFNhbXBsZUZpbGUsIFByb2plY3RPcHRpb25zIH0gZnJvbSBcInByb2plblwiO1xuaW1wb3J0IHsgdHJ5UmVhZEZpbGVTeW5jIH0gZnJvbSBcInByb2plbi9saWIvdXRpbFwiO1xuaW1wb3J0IHsgUGFyc2VkU3BlYyB9IGZyb20gXCIuL2NvbXBvbmVudHMvcGFyc2VkLXNwZWNcIjtcblxuLy8gaW5pdGlhbGl6ZSBsb2dnZXJcbmNvbnN0IGxvZ2dlciA9IGdldExvZ2dlcigpO1xuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIHRoZSBPcGVuQVBJIHNwZWMgcHJvamVjdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9wZW5BcGlTcGVjUHJvamVjdE9wdGlvbnMgZXh0ZW5kcyBQcm9qZWN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgT3BlbkFQSSBzcGVjaWZpY2F0aW9uIGZpbGUuXG4gICAqIEBkZWZhdWx0IFwic3BlYy55YW1sXCJcbiAgICovXG4gIHJlYWRvbmx5IHNwZWNGaWxlTmFtZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBvdXRwdXQgcGFyc2VkIE9wZW5BUEkgc3BlY2lmaWNhdGlvbiBmaWxlLiBNdXN0IGVuZCB3aXRoIC5qc29uLlxuICAgKiBAZGVmYXVsdCBcIi5wYXJzZWQtc3BlYy5qc29uXCJcbiAgICovXG4gIHJlYWRvbmx5IHBhcnNlZFNwZWNGaWxlTmFtZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBkaXJlY3RvcnkgdG8gd3JpdGUgdGhlIHBhcnNlZCBzcGVjIHRvLCByZWxhdGl2ZSB0byB0aGUgcHJvamVjdCBvdXRwdXQgZGlyZWN0b3J5XG4gICAqIEBkZWZhdWx0IHRoZSBwcm9qZWN0IG91dGRpclxuICAgKi9cbiAgcmVhZG9ubHkgcGFyc2VkU3BlY091dGRpcj86IHN0cmluZztcbiAgLyoqXG4gICAqIE92ZXJyaWRlIHRoZSBkaXJlY3Rvcnkgb2YgdGhlIF9pbnB1dF8gT3BlbkFQSSBzcGVjaWZpY2F0aW9uIGFuZCBza2lwIHN5bnRoZXNpemluZyB0aGUgc2FtcGxlIHNwZWMueWFtbFxuICAgKi9cbiAgcmVhZG9ubHkgb3ZlcnJpZGVTcGVjUGF0aD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBQcm9qZWN0IGNvbnRhaW5pbmcgdGhlIE9wZW5BUEkgc3BlYywgYW5kIGEgcGFyc2VkIHNwZWMgZm9yIHVzZSBieSB0aGUgQ0RLIGNvbnN0cnVjdFxuICovXG5leHBvcnQgY2xhc3MgT3BlbkFwaVNwZWNQcm9qZWN0IGV4dGVuZHMgUHJvamVjdCB7XG4gIHB1YmxpYyByZWFkb25seSBzcGVjUGF0aDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcGFyc2VkU3BlY1BhdGg6IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgc3BlY0ZpbGVOYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBwYXJzZWRTcGVjRmlsZU5hbWU6IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgc3BlY0NoYW5nZWQ6IGJvb2xlYW4gPSB0cnVlO1xuXG4gIC8vIFN0b3JlIHdoZXRoZXIgd2UndmUgc3ludGhlc2l6ZWQgdGhlIHByb2plY3RcbiAgcHJpdmF0ZSBzeW50aGVkOiBib29sZWFuID0gZmFsc2U7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogT3BlbkFwaVNwZWNQcm9qZWN0T3B0aW9ucykge1xuICAgIHN1cGVyKG9wdGlvbnMpO1xuICAgIGxvZ2dlci50cmFjZShcIk9wZW5BcGlTcGVjUHJvamVjdCBjb25zdHJ1Y3RvclwiKTtcbiAgICAvLyBIQUNLOiByZW1vdmUgYWxsIGNvbXBvbmVudHMgYnV0IHRoZSBvbmVzIHdlIGFyZSByZWdpc3RlcmluZyAtIHJlbW92ZXMgLmdpdGlnbm9yZSwgdGFza3MsIGV0YyBzaW5jZSB0aGVzZSBhcmVcbiAgICAvLyB1bnVzZWQgYW5kIGEgZGlzdHJhY3Rpb24gZm9yIGVuZC11c2VycyFcbiAgICAvLyBAdHMtaWdub3JlXG4gICAgdGhpcy5fY29tcG9uZW50cyA9IFtdO1xuXG4gICAgdGhpcy5zcGVjRmlsZU5hbWUgPSBvcHRpb25zLnNwZWNGaWxlTmFtZSA/PyBcInNwZWMueWFtbFwiO1xuICAgIHRoaXMucGFyc2VkU3BlY0ZpbGVOYW1lID0gb3B0aW9ucy5wYXJzZWRTcGVjRmlsZU5hbWUgPz8gXCIucGFyc2VkLXNwZWMuanNvblwiO1xuXG4gICAgaWYgKCF0aGlzLnBhcnNlZFNwZWNGaWxlTmFtZS5lbmRzV2l0aChcIi5qc29uXCIpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJQYXJzZWQgc3BlYyBmaWxlIG11c3QgZW5kIHdpdGggLmpzb25cIik7XG4gICAgfVxuXG4gICAgdGhpcy5zcGVjUGF0aCA9XG4gICAgICBvcHRpb25zLm92ZXJyaWRlU3BlY1BhdGggPz8gcGF0aC5qb2luKHRoaXMub3V0ZGlyLCB0aGlzLnNwZWNGaWxlTmFtZSk7XG4gICAgdGhpcy5wYXJzZWRTcGVjUGF0aCA9IHBhdGguam9pbihcbiAgICAgIHRoaXMub3V0ZGlyLFxuICAgICAgLi4uKG9wdGlvbnMucGFyc2VkU3BlY091dGRpciA/IFtvcHRpb25zLnBhcnNlZFNwZWNPdXRkaXJdIDogW10pLFxuICAgICAgdGhpcy5wYXJzZWRTcGVjRmlsZU5hbWVcbiAgICApO1xuXG4gICAgbG9nZ2VyLmRlYnVnKGBzcGVjUGF0aCA9IFwiJHt0aGlzLnNwZWNQYXRofVwiYCk7XG4gICAgbG9nZ2VyLmRlYnVnKGBwYXJzZWRTcGVjUGF0aCA9IFwiJHt0aGlzLnBhcnNlZFNwZWNQYXRofVwiYCk7XG5cbiAgICAvLyBDcmVhdGUgYSBzYW1wbGUgT3BlbkFQSSBzcGVjIHlhbWwgaWYgbm90IGRlZmluZWRcbiAgICBpZiAoIW9wdGlvbnMub3ZlcnJpZGVTcGVjUGF0aCkge1xuICAgICAgbmV3IFNhbXBsZUZpbGUodGhpcywgdGhpcy5zcGVjRmlsZU5hbWUsIHtcbiAgICAgICAgc291cmNlUGF0aDogcGF0aC5qb2luKFxuICAgICAgICAgIF9fZGlybmFtZSxcbiAgICAgICAgICBcIi4uXCIsXG4gICAgICAgICAgXCIuLlwiLFxuICAgICAgICAgIFwiLi5cIixcbiAgICAgICAgICBcInNhbXBsZXNcIixcbiAgICAgICAgICBcInNwZWMueWFtbFwiXG4gICAgICAgICksXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBDaGVjayBpZiB0aGVyZSBpcyBhbHJlYWR5IGEgLnBhcnNlZC1zcGVjLmpzb24gcHJlc2VudFxuICAgIGNvbnN0IGV4aXN0aW5nUGFyc2VkU3BlY0pzb24gPSB0cnlSZWFkRmlsZVN5bmModGhpcy5wYXJzZWRTcGVjUGF0aCk7XG4gICAgaWYgKGV4aXN0aW5nUGFyc2VkU3BlY0pzb24gIT0gbnVsbCkge1xuICAgICAgLy8gZ2VuZXJhdGUgYSBuZXcgdGVtcG9yYXJ5IHBhcnNlZC1zcGVjXG4gICAgICBjb25zdCB0bXBQYXJzZVNwZWNQYXRoID0gYCR7dGhpcy5wYXJzZWRTcGVjUGF0aH0udG1wYDtcbiAgICAgIGxvZ2dlci50cmFjZShgR2VuZXJhdGluZyB0ZW1wIHNwZWMgYXQgXCIke3RtcFBhcnNlU3BlY1BhdGh9XCJgKTtcbiAgICAgIFBhcnNlZFNwZWMucGFyc2UodGhpcy5zcGVjUGF0aCwgdG1wUGFyc2VTcGVjUGF0aCk7XG5cbiAgICAgIC8vIGxvYWQgaXRcbiAgICAgIGNvbnN0IG5ld1BhcnNlZFNwZWNKc29uID0gdHJ5UmVhZEZpbGVTeW5jKHRtcFBhcnNlU3BlY1BhdGgpO1xuXG4gICAgICBpZiAobmV3UGFyc2VkU3BlY0pzb24gIT0gbnVsbCkge1xuICAgICAgICBjb25zdCBwcmV2UGFyc2VTcGVjID0gSlNPTi5wYXJzZShleGlzdGluZ1BhcnNlZFNwZWNKc29uKTtcbiAgICAgICAgY29uc3QgbmV3UGFyc2VkU3BlYyA9IEpTT04ucGFyc2UobmV3UGFyc2VkU3BlY0pzb24pO1xuXG4gICAgICAgIC8vIGNoZWNrIGlmIHNwZWMgY2hhbmdlZCBhbmQga2VlcCBpdFxuICAgICAgICBsb2dnZXIudHJhY2UoXCJDb21wYXJpbmcgcHJldmlvdXMgYW5kIG5ld2x5IGdlbmVyYXRlZCBzcGVjcy5cIik7XG4gICAgICAgIHRoaXMuc3BlY0NoYW5nZWQgPSAhaXNFcXVhbChwcmV2UGFyc2VTcGVjLCBuZXdQYXJzZWRTcGVjKTtcbiAgICAgICAgbG9nZ2VyLmRlYnVnKGBTcGVjIENoYW5nZWQgOjogJHt0aGlzLnNwZWNDaGFuZ2VkfWApO1xuXG4gICAgICAgIC8vIHJlbW92ZSB0bXAgcGFyc2VkLXNwZWMgZmlsZVxuICAgICAgICBsb2dnZXIudHJhY2UoXCJSZW1vdmluZyB0ZW1wIHNwZWMgZmlsZS5cIik7XG4gICAgICAgIHVubGlua1N5bmModG1wUGFyc2VTcGVjUGF0aCk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGxvZ2dlci5kZWJ1ZyhgTm8gcGFyc2VkU3BlYyBmaWxlIGZvdW5kIGF0IFwiJHt0aGlzLnBhcnNlZFNwZWNQYXRofVwiLmApO1xuICAgIH1cblxuICAgIC8vIFBhcnNlIHRoZSBzcGVjIHRvIHByb2R1Y2UgYSBjb25zb2xpZGF0ZWQsIGJ1bmRsZWQgc3BlYyB3aGljaCBjYW4gYmUgcmVhZCBieSBjZGsgY29uc3RydWN0cyBvciBvdGhlciB0b29saW5nXG4gICAgbmV3IFBhcnNlZFNwZWModGhpcywge1xuICAgICAgc3BlY1BhdGg6IHRoaXMuc3BlY1BhdGgsXG4gICAgICBvdXRwdXRQYXRoOiB0aGlzLnBhcnNlZFNwZWNQYXRoLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbmhlcml0RG9jXG4gICAqL1xuICBzeW50aCgpIHtcbiAgICAvLyBTYXZlIHNvbWUgdGltZSBieSBvbmx5IHN5bnRoZXNpemluZyBvbmNlLiBXZSBzeW50aGVzaXplIHRoaXMgcHJvamVjdCBlYXJseSBzbyB0aGF0IGl0J3MgYXZhaWxhYmxlIGZvciB0aGUgcGFyZW50XG4gICAgLy8gcHJvamVjdCdzIGluc3RhbGwgcGhhc2UgKHByZS1zeW50aCkuIFByb2plbiB3aWxsIGNhbGwgdGhpcyBtZXRob2QgYWdhaW4gYXQgdGhlIHVzdWFsIHRpbWUgdG8gc3ludGhlc2l6ZSB0aGlzLFxuICAgIC8vIHByb2plY3QsIGF0IHdoaWNoIHBvaW50IHdlJ3JlIGFscmVhZHkgZG9uZSBzbyBjYW4gc2tpcC5cbiAgICBpZiAodGhpcy5zeW50aGVkKSB7XG4gICAgICBsb2dnZXIudHJhY2UoXCJPcGVuQXBpU3BlY1Byb2plY3QgYWxyZWFkeSBzeW50aGVkLiBTa2lwcGluZy4uLlwiKTtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgbG9nZ2VyLnRyYWNlKFwiT3BlbkFwaVNwZWNQcm9qZWN0IHN5bnRoXCIpO1xuICAgIHN1cGVyLnN5bnRoKCk7XG5cbiAgICB0aGlzLnN5bnRoZWQgPSB0cnVlO1xuICB9XG59XG4iXX0=