"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParsedSpec = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
const logger = (0, log4js_1.getLogger)();
/**
 * Component for parsing the yaml OpenAPI spec as a single json object, resolving references etc.
 */
class ParsedSpec extends projen_1.Component {
    static parse(specPath, outputPath) {
        // Parse the spec and write to the target output path
        (0, util_1.exec)(`./parse-openapi-spec --specPath=${specPath} --outputPath=${outputPath}`, {
            cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "parser"),
        });
    }
    constructor(project, options) {
        super(project);
        this.options = options;
    }
    synthesize() {
        super.synthesize();
        ParsedSpec.parse(this.options.specPath, this.options.outputPath);
        const singleSpecFile = (0, util_1.tryReadFileSync)(this.options.outputPath);
        if (!singleSpecFile) {
            throw new Error(`Failed to parse specification at ${this.options.specPath}`);
        }
        const parsedSpec = JSON.parse(singleSpecFile);
        // To avoid duplicating custom generated code (eg. OperationConfig or handler wrappers) and causing build errors, we
        // will apply the OpenAPI Normalizer to KEEP_ONLY_FIRST_TAG_IN_OPERATION when generating code. Tags are still
        // preserved in the specification to allow for better documentation.
        // See: https://github.com/OpenAPITools/openapi-generator/pull/14465
        const operationsWithMultipleTags = Object.entries(parsedSpec.paths).flatMap(([urlPath, methods]) => Object.entries(methods ?? {})
            .filter(([, operation]) => typeof operation === "object" &&
            "tags" in operation &&
            operation.tags &&
            operation.tags.length > 1)
            .map(([httpMethod]) => `${httpMethod} ${urlPath}`));
        if (operationsWithMultipleTags.length > 0) {
            logger.warn(`The following operations had multiple tags: ${operationsWithMultipleTags.join(", ")}. Code will only be generated for each operation's first tag.`);
        }
    }
}
exports.ParsedSpec = ParsedSpec;
//# sourceMappingURL=data:application/json;base64,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