"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPythonSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample python source code
 */
const getPythonSampleSource = (options) => ({
    "__init__.py": "#",
    // This file provides a type-safe interface to the exported OpenApiGatewayRestApi construct
    "api.py": `from dataclasses import fields
from ${options.openApiGatewayPackageName} import OpenApiGatewayRestApi, OpenApiIntegration
from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import OperationLookup, OperationConfig
from ${options.moduleName}.spec_utils import SPEC, SPEC_PATH

class Api(OpenApiGatewayRestApi):
    """
    Type-safe construct for the API Gateway resources defined by the spec.
    You will likely not need to modify this file, and can instead extend it and define your integrations.
    """
    def __init__(self, scope, id, integrations: OperationConfig[OpenApiIntegration], **kwargs):
        super().__init__(scope, id,
            **kwargs,
            integrations={ field.name: getattr(integrations, field.name) for field in fields(integrations) },
            spec=SPEC,
            spec_path=SPEC_PATH,
            operation_lookup=OperationLookup,
        )
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample_api.py": `from ${options.openApiGatewayPackageName} import Authorizers, Integrations, OpenApiIntegration
from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import OperationConfig
from aws_cdk.aws_lambda import LayerVersion, Code, Function, Runtime
from .api import Api
from constructs import Construct
from ${options.moduleName}.spec_utils import get_generated_client_layer_directory
from pathlib import Path
from os import path

class SampleApi(Construct):
  """
  An example of how to add integrations to your api
  """
  def __init__(self, scope, id):
      super().__init__(scope, id)

      # Layer which contains the generated client.
      self.generated_client_layer = LayerVersion(self, 'GeneratedClientLayer',
          code=Code.from_asset(get_generated_client_layer_directory())
      )

      self.api = Api(self, 'Api',
          default_authorizer=Authorizers.iam(),
          integrations=OperationConfig(
              say_hello=OpenApiIntegration(
                  integration=Integrations.lambda_(Function(self, 'SayHello',
                      runtime=Runtime.PYTHON_3_9,
                      code=Code.from_asset(path.join(str(Path(__file__).parent.absolute()), 'handlers')),
                      handler="say_hello_handler_sample.handler",
                      layers=[self.generated_client_layer],
                  )),
              ),
          ),
      )
`,
            // Generate an example lambda handler
            "handlers/say_hello_handler_sample.py": `from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse, SayHelloOperationResponses
from ${options.pythonClientPackageName}.model.say_hello_response_content import SayHelloResponseContent

@say_hello_handler
def handler(input: SayHelloRequest, **kwargs) -> SayHelloOperationResponses:
    """
    An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs
    """
    return ApiResponse(
        status_code=200,
        body=SayHelloResponseContent(message="Hello {}!".format(input.request_parameters["name"])),
        headers={}
    )
`,
        }
        : {}),
});
exports.getPythonSampleSource = getPythonSampleSource;
//# sourceMappingURL=data:application/json;base64,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