"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getJavaSampleSource = void 0;
const getJavaSampleSource = (options) => {
    const apiPackage = options.apiSrcDir.split("/").join(".");
    return {
        "ApiProps.java": `package ${apiPackage};

import software.amazon.awscdk.services.apigateway.CorsOptions;
import software.amazon.awscdk.services.apigateway.StageOptions;
import software.amazon.awscdk.services.apigateway.RestApiBaseProps;
import software.amazon.awscdk.services.apigateway.DomainNameOptions;
import software.amazon.awscdk.services.apigateway.EndpointType;
import software.amazon.awscdk.services.iam.PolicyDocument;
import software.aws.awsprototypingsdk.openapigateway.Authorizer;
import software.aws.awsprototypingsdk.openapigateway.OpenApiIntegration;

import ${options.javaClientPackage}.api.OperationConfig;

import java.util.List;
import java.util.Map;

/**
 * Properties for the Api construct
 */
@lombok.Builder @lombok.Getter
public class ApiProps implements RestApiBaseProps {
    public OperationConfig<OpenApiIntegration> integrations;
    public Authorizer defaultAuthorizer;
    public CorsOptions corsOptions;

    // Rest API Props
    public Boolean cloudWatchRole;
    public Boolean deploy;
    public StageOptions deployOptions;
    public String description;
    public Boolean disableExecuteApiEndpoint;
    public DomainNameOptions domainName;
    public String endpointExportName;
    public List<EndpointType> endpointTypes;
    public Boolean failOnWarnings;
    public Map<String, String> parameters;
    public PolicyDocument policy;
    public String restApiName;
    public Boolean retainDeployments;
}
`,
        "Api.java": `package ${apiPackage};

import com.fasterxml.jackson.databind.ObjectMapper;
import software.aws.awsprototypingsdk.openapigateway.MethodAndPath;
import software.aws.awsprototypingsdk.openapigateway.OpenApiGatewayRestApi;
import software.aws.awsprototypingsdk.openapigateway.OpenApiGatewayRestApiProps;
import software.constructs.Construct;
import ${options.javaClientPackage}.api.OperationLookup;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
public class Api extends OpenApiGatewayRestApi {
    private static class SpecDetails {
        static String specPath;
        static Object spec;

        static {
            try {
                // The parsed spec is included as a packaged resource
                URL specUrl = SpecDetails.class.getClassLoader().getResource("${options.specDir}/${options.parsedSpecFileName}");

                // We'll write the parsed spec to a temporary file outside of the jar to ensure CDK can package it as an asset
                Path parsedSpecPath = Files.createTempFile("parsed-spec", ".json");
                specPath = parsedSpecPath.toString();

                ObjectMapper json = new ObjectMapper();
                spec = json.readValue(specUrl, Object.class);
                json.writeValue(parsedSpecPath.toFile(), spec);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public Api(Construct scope, String id, ApiProps props) {
        super(scope, id, OpenApiGatewayRestApiProps.builder()
                .defaultAuthorizer(props.getDefaultAuthorizer())
                .corsOptions(props.getCorsOptions())
                .operationLookup(OperationLookup.getOperationLookup()
                        .entrySet()
                        .stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, e -> MethodAndPath.builder()
                                .method(e.getValue().get("method"))
                                .path(e.getValue().get("path"))
                                .build())))
                .spec(SpecDetails.spec)
                .specPath(SpecDetails.specPath)
                .integrations(props.getIntegrations().asMap())
                // Rest API Base Props
                .cloudWatchRole(props.getCloudWatchRole())
                .deploy(props.getDeploy())
                .deployOptions(props.getDeployOptions())
                .description(props.getDescription())
                .disableExecuteApiEndpoint(props.getDisableExecuteApiEndpoint())
                .domainName(props.getDomainName())
                .endpointExportName(props.getEndpointExportName())
                .endpointTypes(props.getEndpointTypes())
                .failOnWarnings(props.getFailOnWarnings())
                .parameters(props.getParameters())
                .policy(props.getPolicy())
                .restApiName(props.getRestApiName())
                .retainDeployments(props.getRetainDeployments())
                .build());
    }
}
`,
        ...(options.sampleCode !== false
            ? {
                "SayHelloHandler.java": `package ${apiPackage};

import ${options.javaClientPackage}.api.Handlers.SayHello;
import ${options.javaClientPackage}.api.Handlers.SayHello200Response;
import ${options.javaClientPackage}.api.Handlers.SayHelloRequestInput;
import ${options.javaClientPackage}.api.Handlers.SayHelloResponse;
import ${options.javaClientPackage}.model.SayHelloResponseContent;

/**
 * An example lambda handler which uses the generated handler wrapper class (Handlers.SayHello) to manage marshalling
 * inputs and outputs.
 */
public class SayHelloHandler extends SayHello {
    @Override
    public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
        return SayHello200Response.of(SayHelloResponseContent.builder()
                .message(String.format("Hello %s", sayHelloRequestInput.getInput().getRequestParameters().getName()))
                .build());
    }
}
`,
                "SampleApi.java": `package ${apiPackage};

import software.amazon.awscdk.Duration;
import software.amazon.awscdk.services.apigateway.CorsOptions;
import software.amazon.awscdk.services.lambda.Code;
import software.amazon.awscdk.services.lambda.Function;
import software.amazon.awscdk.services.lambda.FunctionProps;
import software.amazon.awscdk.services.lambda.Runtime;
import software.aws.awsprototypingsdk.openapigateway.Authorizers;
import software.aws.awsprototypingsdk.openapigateway.Integrations;
import software.aws.awsprototypingsdk.openapigateway.OpenApiIntegration;
import ${options.javaClientPackage}.api.OperationConfig;
import software.constructs.Construct;

import java.net.URISyntaxException;
import java.util.Arrays;

/**
 * An example of how to wire lambda handler functions to API operations
 */
public class SampleApi extends Api {
    public SampleApi(Construct scope, String id) {
        super(scope, id, ApiProps.builder()
                .defaultAuthorizer(Authorizers.iam())
                .corsOptions(CorsOptions.builder()
                        .allowOrigins(Arrays.asList("*"))
                        .allowMethods(Arrays.asList("*"))
                        .build())
                .integrations(OperationConfig.<OpenApiIntegration>builder()
                        .sayHello(OpenApiIntegration.builder()
                                .integration(Integrations.lambda(
                                        new Function(scope, "say-hello", FunctionProps.builder()
                                                // Use the entire project jar for the lambda code in order to provide a simple,
                                                // "one-click" way to build the api. However this jar is much larger than necessary
                                                // since it includes cdk infrastructure, dependencies etc.
                                                // It is recommended to follow the instructions in the "Java API Lambda Handlers"
                                                // section of the open-api-gateway README to define your lambda handlers as a
                                                // separate project.
                                                .code(Code.fromAsset(SampleApi.getJarPath()))
                                                .handler("${apiPackage}.SayHelloHandler")
                                                .runtime(Runtime.JAVA_11)
                                                .timeout(Duration.seconds(30))
                                                .build())))
                                .build())
                        .build())
                .build());
    }

    private static String getJarPath() {
        try {
            // Retrieve the path of the jar in which this class resides
            return SampleApi.class.getProtectionDomain().getCodeSource().getLocation().toURI().getPath();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }
}

`,
            }
            : {}),
    };
};
exports.getJavaSampleSource = getJavaSampleSource;
//# sourceMappingURL=data:application/json;base64,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