"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 * @deprecated Please use TypeSafeApiProject instead. This will be removed in the PDK GA 1.0 release.
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...projectOptions?.venvOptions,
            },
            pip: true,
            poetry: false,
            // No tests by default, but allow user to override
            pytest: projectOptions.pytest ?? false,
            setuptools: true,
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib", "cdk-nag"]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.PYTHON,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = (0, generate_1.generateClientProjects)(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated python client is built first
                this.parent.addImplicitDependency(this, this.generatedPythonClient);
            }
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in spec_utils.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "spec_utils.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC_PATH = path.join(str(Path(__file__).absolute().parent), "${this.specDir}/${spec.parsedSpecFileName}")
SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: (0, python_2.getPythonSampleSource)(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than python project options.
     */
    preConstruct(options) {
        return options;
    }
}
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.19.36" };
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
//# sourceMappingURL=data:application/json;base64,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