"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const java_2 = require("./samples/java");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "software.aws.awsprototypingsdk/open-api-gateway@^0";
/**
 * Synthesizes a Java Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-java
 * @deprecated Please use TypeSafeApiProject instead. This will be removed in the PDK GA 1.0 release.
 */
class OpenApiGatewayJavaProject extends java_1.JavaProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sample: false,
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "software.constructs/constructs@^10",
            "software.amazon.awscdk/aws-cdk-lib@^2",
            "io.github.cdklabs/cdknag@^2",
            "org.projectlombok/lombok@^1",
            "com.fasterxml.jackson.core/jackson-databind@^2",
            "io.github.cdklabs/projen@^0",
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep.split("@")[0], projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Synthesize the openapi spec early since it's used by the generated java client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join("src", this.specDir),
            // Write the parsed spec to the resources directory so that it can be packaged into the jar
            parsedSpecOutdir: path.join(...this.specDir.split("/").map(() => ".."), "main", "resources", this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the java client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.JAVA);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.JAVA,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        this.generatedClients = (0, generate_1.generateClientProjects)(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedJavaClient = this.generatedClients[languages_1.ClientLanguage.JAVA];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedJavaClient.synth();
        // Add a dependency on the generated java client
        this.addDependency(`${this.generatedJavaClient.pom.groupId}/${this.generatedJavaClient.pom.artifactId}@${this.generatedJavaClient.pom.version}`);
        // Add a dependency on the generated java client repository
        this.pom.addRepository({
            url: `file://./${this.generatedCodeDir}/${languages_1.ClientLanguage.JAVA}/dist/java`,
            id: "generated-java-api-client",
        });
        if (this.hasParent) {
            // Since the generated java client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated java client is built first
                this.parent.addImplicitDependency(this, this.generatedJavaClient);
            }
        }
        // We build this project as a super jar so that it can be deployed as a lambda. This isn't ideal for jar size and
        // so is not the recommended approach, however is makes it possible for a "one-click" way to get started with a
        // full hello world api. Included in the generated "SampleApi.java" is a comment encouraging users to read the
        // README and to move to defining a separate package for lambda handlers.
        this.pom.addPlugin("org.apache.maven.plugins/maven-shade-plugin@3.3.0", {
            configuration: {
                createDependencyReducedPom: false,
                // Name is the same as the regular maven build jar, which ensures the api jar path resolves to the super jar
                // for deployment as a lambda.
                finalName: `${this.name}-${options.version}`,
            },
            executions: [
                {
                    id: "shade-task",
                    phase: "package",
                    goals: ["shade"],
                },
            ],
        });
        const javaClientPackage = `${this.generatedJavaClient.pom.groupId}.${this.generatedJavaClient.name}.client`;
        // Generate the sample source code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            sampleCode: options.sample ?? true,
            apiSrcDir: this.apiSrcDir,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            javaClientPackage,
        };
        new projen_1.SampleDir(this, path.join("src", "main", "java", this.apiSrcDir), {
            files: (0, java_2.getJavaSampleSource)(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than java project options.
     */
    preConstruct(options) {
        return options;
    }
}
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayJavaProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayJavaProject", version: "0.19.36" };
exports.OpenApiGatewayJavaProject = OpenApiGatewayJavaProject;
//# sourceMappingURL=data:application/json;base64,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