"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptClientProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const generated_typescript_client_source_code_1 = require("./components/generated-typescript-client-source-code");
const open_api_generator_ignore_file_1 = require("./components/open-api-generator-ignore-file");
/**
 * Typescript project containing a typescript client (and lambda handler wrappers) generated using OpenAPI Generator CLI
 */
class GeneratedTypescriptClientProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
            // Disable tests unless explicitly enabled
            jest: options.jest ?? false,
        });
        // Store whether we've synthesized the project
        this.synthed = false;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmConfig = new javascript_1.NpmConfig(this);
            this.npmConfig.addConfig("strict-peer-dependencies", "false");
        }
        // For event and context types
        this.addDeps("@types/aws-lambda");
        // Tell OpenAPI Generator CLI not to generate files that we will generate via this project, or don't need.
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("package.json", "tsconfig.json", "tsconfig.esm.json", "src/index.ts");
        new generated_typescript_client_source_code_1.GeneratedTypescriptClientSourceCode(this, {
            specPath: options.specPath,
            invokeGenerator: options.generateClient,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.GeneratedTypescriptClientProject = GeneratedTypescriptClientProject;
//# sourceMappingURL=data:application/json;base64,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