"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonClientProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const generated_python_client_source_code_1 = require("./components/generated-python-client-source-code");
const open_api_generator_ignore_file_1 = require("./components/open-api-generator-ignore-file");
/**
 * Python project containing a python client (and lambda handler wrappers) generated using OpenAPI Generator CLI
 */
class GeneratedPythonClientProject extends python_1.PythonProject {
    constructor(options) {
        super({
            sample: false,
            pytest: false,
            ...options,
        });
        /**
         * The directory in which the built layer is output
         */
        this.layerDistDir = "dist/layer";
        // Store whether we've synthesized the project
        this.synthed = false;
        // With pip and venv (default), it's useful to install our package into the shared venv to make
        // it easier for other packages in the monorepo to take dependencies on this package.
        if ((options.venv ?? true) && (options.pip ?? true)) {
            this.depsManager.installTask.exec("pip install --editable .");
        }
        // Package into a directory that can be used as a lambda layer. This is done as part of install since the end user
        // must control build order in the monorepo via explicit dependencies, and adding here means we can run as part of
        // initial project synthesis which ensures this is created regardless of whether the user has remembered to
        // configure build order.
        if (options.generateLayer) {
            const relativeLayerDir = path.join(".", this.layerDistDir, "python");
            this.depsManager.installTask.exec(`rm -rf ${relativeLayerDir}`);
            this.depsManager.installTask.exec(`pip install . --target ${relativeLayerDir}`);
        }
        // Use a package.json to ensure the client is discoverable by nx
        new projen_1.JsonFile(this, "package.json", {
            obj: {
                name: this.name,
                __pdk__: true,
                version: options.version,
                scripts: Object.fromEntries(this.tasks.all.map((task) => [task.name, `npx projen ${task.name}`])),
            },
            readonly: true,
        });
        new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        new generated_python_client_source_code_1.GeneratedPythonClientSourceCode(this, {
            specPath: options.specPath,
            invokeGenerator: options.generateClient,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.GeneratedPythonClientProject = GeneratedPythonClientProject;
//# sourceMappingURL=data:application/json;base64,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