"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateClientProjects = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const generated_java_client_project_1 = require("./generated-java-client-project");
const generated_python_client_project_1 = require("./generated-python-client-project");
const generated_typescript_client_project_1 = require("./generated-typescript-client-project");
const languages_1 = require("../languages");
const logger = (0, log4js_1.getLogger)();
/**
 * Returns a generated client project for the given language
 */
const generateClientProject = (languageConfig, options) => {
    switch (languageConfig.clientLanguage) {
        case languages_1.ClientLanguage.TYPESCRIPT: {
            logger.trace("Attempting to generate TYPESCRIPT client project.");
            return new generated_typescript_client_project_1.GeneratedTypescriptClientProject({
                parent: options.parent,
                generateClient: languageConfig.generate,
                // Ensure kebab-case for typescript
                name: `${options.parentPackageName}-${languages_1.ClientLanguage.TYPESCRIPT}`.replace(/_/g, "-"),
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.TYPESCRIPT),
                specPath: options.parsedSpecPath,
                ...options.typescriptOptions,
            });
        }
        case languages_1.ClientLanguage.PYTHON: {
            logger.trace("Attempting to generate PYTHON client project.");
            // Ensure snake_case for python
            const moduleName = `${options.parentPackageName}_${languages_1.ClientLanguage.PYTHON}`
                .replace(/@/g, "")
                .replace(/[\-/]/g, "_");
            return new generated_python_client_project_1.GeneratedPythonClientProject({
                parent: options.parent,
                generateClient: languageConfig.generate,
                // Use dashes in project name since distributable's PKG-INFO always converts _ to -
                // https://stackoverflow.com/questions/36300788/python-package-wheel-pkg-info-name
                name: moduleName.replace(/_/g, "-"),
                moduleName,
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.PYTHON),
                specPath: options.parsedSpecPath,
                ...options.pythonOptions,
            });
        }
        case languages_1.ClientLanguage.JAVA: {
            logger.trace("Attempting to generate JAVA client project.");
            // Ensure no dashes/underscores since name is used in package name
            const javaProjectName = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[\-/_]/g, "");
            const artifactId = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[/_]/g, "-");
            return new generated_java_client_project_1.GeneratedJavaClientProject({
                parent: options.parent,
                generateClient: languageConfig.generate,
                name: javaProjectName,
                artifactId,
                groupId: "com.generated.api",
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.JAVA),
                specPath: options.parsedSpecPath,
                ...options.javaOptions,
            });
        }
        default:
            throw new Error(`Unknown client language ${languageConfig.clientLanguage}`);
    }
};
/**
 * Generate API clients in the given languages
 * @param languages the languages to generate clients for
 * @param options options for the projects to be created
 */
const generateClientProjects = (languageConfigs, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Clients",
            "",
            "This directory contains generated client code based on your OpenAPI Specification file (spec.yaml).",
            "",
            "Like other `projen` managed files, this directory should be checked in to source control, but should not be edited manually.",
        ],
        readonly: true,
    });
    const generatedClients = {};
    languageConfigs.forEach((languageConfig) => {
        const project = generateClientProject(languageConfig, options);
        if (project != null) {
            generatedClients[languageConfig.clientLanguage] = project;
        }
    });
    return generatedClients;
};
exports.generateClientProjects = generateClientProjects;
//# sourceMappingURL=data:application/json;base64,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