"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeCustomDocsGenerator = exports.invokeOpenApiGenerator = exports.NonClientGeneratorDirectory = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const util_1 = require("projen/lib/util");
/**
 * Enum for generator directories for non-client generators
 */
var NonClientGeneratorDirectory;
(function (NonClientGeneratorDirectory) {
    NonClientGeneratorDirectory["DOCS"] = "docs";
})(NonClientGeneratorDirectory = exports.NonClientGeneratorDirectory || (exports.NonClientGeneratorDirectory = {}));
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Clean up any files that have already been generated by the openapi generator
 * @param generatedOutputPath output path for the generator
 */
const cleanPreviouslyGeneratedFiles = (generatedOutputPath) => {
    // OpenAPI generator writes a manifest called FILES which lists the files it generated.
    const openApiGeneratedFilesManifestPath = path.join(generatedOutputPath, ".openapi-generator", "FILES");
    // If the manifest exists, delete the files it lists
    if (fs.existsSync(openApiGeneratedFilesManifestPath)) {
        const previouslyGeneratedFiles = new Set(fs
            .readFileSync(openApiGeneratedFilesManifestPath, { encoding: "utf-8" })
            .split("\n")
            .filter((x) => x));
        previouslyGeneratedFiles.forEach((previouslyGeneratedFile) => {
            fs.unlinkSync(path.join(generatedOutputPath, previouslyGeneratedFile));
        });
    }
};
/**
 * Generate client code or docs by invoking the root generate script
 */
const invokeOpenApiGenerator = (options) => {
    // Prior to generating, clean up any previously generated files so that we don't have stale generated code from
    // previous executions.
    cleanPreviouslyGeneratedFiles(options.outputPath);
    const srcDir = options.srcDir ?? "src";
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    const normalizers = options.normalizers
        ? ` --openapi-normalizer "${serializeProperties(Object.fromEntries(Object.entries(options.normalizers).map(([k, v]) => [k, `${v}`])))}"`
        : "";
    (0, util_1.exec)(`./generate --generator ${options.generator} --spec-path ${options.specPath} --output-path ${options.outputPath} --generator-dir ${options.generatorDirectory} --src-dir ${srcDir}${additionalProperties}${normalizers}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "generators"),
    });
};
exports.invokeOpenApiGenerator = invokeOpenApiGenerator;
/**
 * Invoke a custom documentation generator script
 */
const invokeCustomDocsGenerator = (options) => {
    (0, util_1.exec)(`./${options.generator}${options.args ? ` ${options.args}` : ""}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "custom", "docs"),
    });
};
exports.invokeCustomDocsGenerator = invokeCustomDocsGenerator;
//# sourceMappingURL=data:application/json;base64,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