"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientSettings = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
// initialize logger
const logger = (0, log4js_1.getLogger)();
/**
 * Component for parsing the yaml OpenAPI spec as a single json object, resolving references etc.
 */
class ClientSettings extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.options = options;
        this.clientSettingsFilename =
            options.clientSettingsFilename ?? ".client-settings.json";
        this.clientSettingsPath = path.join(options.generatedCodeDir, this.clientSettingsFilename);
        // load previously generated client config
        const clientSettingsPathAbs = path.join(project.outdir, this.clientSettingsPath);
        logger.trace(`Reading client settings from ${clientSettingsPathAbs}`);
        const existingClientConfig = (0, util_1.tryReadFileSync)(clientSettingsPathAbs);
        const prevClientLanguages = new Set();
        const prevDocFormats = new Set();
        if (existingClientConfig) {
            const parsedClientConfig = JSON.parse(existingClientConfig);
            // previously generated client settings
            parsedClientConfig.clientLanguages.map((l) => prevClientLanguages.add(l));
            parsedClientConfig.documentationFormats.map((d) => prevDocFormats.add(d));
        }
        this.clientLanguageConfigs = options.clientLanguages.map((clientLanguage) => ({
            clientLanguage,
            isDefault: clientLanguage === options.defaultClientLanguage,
            generate: options.specChanged ||
                options.forceGenerateCodeAndDocs ||
                !prevClientLanguages.has(clientLanguage),
        }));
        this.documentationFormatConfigs = options.documentationFormats.map((documentationFormat) => ({
            documentationFormat,
            generate: options.specChanged ||
                options.forceGenerateCodeAndDocs ||
                !prevDocFormats.has(documentationFormat),
        }));
    }
    synthesize() {
        // store the client config
        // this helps to optimize to re-generate projects when new language/doc format is
        // introduced while the spec doesn't change
        logger.trace(`Generating settings file to ${this.clientSettingsPath}`);
        const clientSettingsJsonFile = new projen_1.JsonFile(this.project, this.clientSettingsPath, {
            obj: {
                clientLanguages: this.options.clientLanguages,
                documentationFormats: this.options.documentationFormats,
            },
            marker: true,
            editGitignore: false,
        });
        logger.debug(`Generated settings file to ${clientSettingsJsonFile.absolutePath}`);
    }
}
exports.ClientSettings = ClientSettings;
//# sourceMappingURL=data:application/json;base64,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